#!/bin/sh
# Copyright © 2015 - present Lenovo.  All rights reserved.
# Confidential and Proprietary.
# Copyright:    IBM Corporation
# Author:       harrigan@us.ibm.com
# Version:      0.1.3

# Change History
# 0.1.0     Alpha Version
# 0.1.1     Beta 1 Version
# 0.1.1     Beta 2 Uses 10 minute timeout
# 0.1.2     add line option " --stop"
# 0.1.3     check c-state dd process before bringing up enable c-state fix

# This script will enable a workaround for a C-State issue found under specific configurations

# Enable/Disable bash debug ...
#set -x

# wangliang workaroud for spawnprocess issue: spawnprocess will remove /sbin/ from defaul list
PATH=$PATH:/sbin/

# Set our core variables ...
CSTATE_FIX_PS="dd if=/dev/zero of=/dev/null"
CSTATE_FIX_CMD="nice -n 19 $CSTATE_FIX_PS"
CSTATE_AFFECTED_CPU_LIST="E5520 E5530 E5540"

#############################################
# THIS SECTION COPIED FROM CDC_INTERFACE.SH #
#############################################

IS_XEN=1 #Set to false by default
VMWARE_4=1 #Set to false by default


# Detect which Linux distribution we are running, in the case of RHEL 4 set CDC_DRIVER_NAME to "usbnet"
detect_os_variant() {

    if [ /etc/redhat-release -nt /etc/vmware-release ]
    then
        VER=`sed "s/.*release //" /etc/redhat-release | awk '{print $1}'|cut -c 1`
        MAJOR_OS=RHEL
        if [ $VER -ge 5 ]
        then
            REL=`sed "s/.*release //" /etc/redhat-release | awk '{print $1}'|cut -c 3`
            CDC_DRIVER_NAME=cdc_ether
        fi
        if [ $VER -le 4 ]
        then
            REL=`awk '{print $10}' /etc/redhat-release | cut -c 1`
            CDC_DRIVER_NAME=usbnet
        fi
        echo "RedHat Enterprise Linux Version $VER Update $REL found ..." >&1
    elif [ -f /etc/SuSE-release ]
    then
        VER=`awk 'NR==2{print $3}' /etc/SuSE-release`
        REL=`awk 'NR==3{print $3}' /etc/SuSE-release`
        CDC_DRIVER_NAME=cdc_ether
        MAJOR_OS=SLES
        echo "SuSE Linux Enterprise Server Version $VER Service Pack $REL found ..." >&1
        if [ $VER -ge 11 ]
            then
                #After the bridged connections are set up in XEN, they persist to non-XEN SLES 11.
                #Because of this, we need to check for a bridged connection to the cdc interface
                #before we attempt to set the usb0.
                IS_XEN=0
        fi
        SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/sysconfig/network"
    elif [ `vmware -v | awk '{print $1}'` == "VMware" ]
    then
        #Since vmware -v output has changed between releases need to detect the release differently
        VER3=`vmware -v | awk '{print $4}' | cut -c 1`
        VER4=`vmware -v | awk '{print $3}' | cut -c 1`
        if [ $VER3 == 3 ]
        then
            VER=3
            REL=`vmware -v | awk '{print $4}' | cut -c 3`
        elif [ $VER4 == 4 ]
        then
            VER=4
            REL=`vmware -v | awk '{print $3}' | cut -c 3`
        fi

        # Set MAJOR_OS
        MAJOR_OS=VMWARE

        # Load correct driver
        if [ $VER == 3 ]
        then
            #Driver does not exist in VMware ESX Server 3.5 prior to U4
            CDC_DRIVER_NAME=CDCEther
        fi
        if [ $VER == 4 ]
        then
            CDC_DRIVER_NAME=cdc_ether
            VMWARE_4=0
        fi
        echo "VMware ESX Server $VER found ..." >&1
    else
        echo "Unable to detect operating system, exiting." >&1
        exit 172
    fi
}
###############################################
# END OF SECTION COPIED FROM CDC_INTERFACE.SH #
###############################################


check_for_cstate_support() {
    # RHEL 5.3 and later has support for C3 State
    if [ "$MAJOR_OS" == "RHEL" -a $VER -eq 5 -a $REL -ge 3 ]
    then
        return 0
    fi
    # RHEL 6 Base has support for C3 State
    if [ "$MAJOR_OS" == "RHEL" -a $VER -ge 6 ]
    then
        return 0
    fi
    # SLES 10 SP3 has support for C3 State
    if [ "$MAJOR_OS" == "SLES" -a $VER -eq 10 -a $REL -ge 2 ]
    then
        return 0
    fi
    # SLES 11 Base has support for C3 State
    if [ "$MAJOR_OS" == "SLES" -a $VER -ge 11 ]
    then
        return 0
    fi
    return 1
}

check_for_32_bit_non_xen_os() {
    PROCESSOR_ARCH=`uname -p`
    if [ "$PROCESSOR_ARCH" == "i686" ] && ( `uname -r | grep -v -q -e '[Xx][Ee][Nn]'` )
    then
        echo "i686 non-xen platform detected ..."
        return 0
    fi
    return 1
}

check_for_cstate_3_enabled() {
    ###########################################################
    # Check /proc/acpi/processor/???0/power for a type[C3]    #
    # Power State.  You can't just check for C3 because       #
    # Linux assigns it's own C# to the power states and       #
    # Linux C3 simply means there a 3 power states supported. #
    ###########################################################
    if [ -f /proc/acpi/processor/0/power ]
    then
        if ( `cat /proc/acpi/processor/0/power | grep -q -e 'type\[C3\]'` )
        then
            echo "ACPI C3 power state is supported and enabled ..."
            return 0
        fi
    fi
    if [ -f /proc/acpi/processor/C000/power ]
    then
        if ( `cat /proc/acpi/processor/C000/power | grep -q -e 'type\[C3\]'` )
        then
            echo "ACPI C3 power state is supported and enabled ..."
            return 0
        fi
    fi
    if [ -f /proc/acpi/processor/CPU0/power ]
    then
        if ( `cat /proc/acpi/processor/CPU0/power | grep -q -e 'type\[C3\]'` )
        then
            echo "ACPI C3 power state is supported and enabled ..."
            return 0
        fi
    fi
    return 1
}

check_for_affected_cpu() {
    PROCESSOR_NAME=`cat /proc/cpuinfo | grep -m 1 -e 'model name' | awk -F ' ' '{print $7}'`
    for CSTATE_AFFECTED_CPU in ${CSTATE_AFFECTED_CPU_LIST};
    do
        if [ "$PROCESSOR_NAME" == "$CSTATE_AFFECTED_CPU" ]
        then
            return 0
        fi
    done
    return 1
}

check_for_single_cpu() {
    # check /proc/cpuinfo for any physical id that is not zero.
    if ( ! `cat /proc/cpuinfo | grep -e 'physical id' | grep -v -q -e '0'` )
    then
        echo "Single $PROCESSOR_NAME CPU package installed ..."
        return 0
    fi
    return 1
}

check_cstatefix_running() {
# check whether any c-state ProcessID already running.
# return 1 if running in proc; else 0.
FOUND=`ps au | grep -m 1 -e 'dd if /dev/zero of /dev/null' | grep -v 'grep'`
if [ "$FOUND" == "" ]
then
return 0
fi

return 1
}

disable_cstate_fix() {
# find and kill c-state processID
	kill `ps au | grep -m 1 -e 'dd if /dev/zero of /dev/null' | awk -F ' ' '{print $2}'`
return 0
}
enable_cstate_fix() {
    #########################################################
    # if C-States are supported and C3 state is enabled     #
    # in 32 bit non-xen RHEL or SLES with a single affected #
    # CPU (E5520 E5530 E5540) installed, then start a       #
    # busyloop to keep the USB-LAN connection active        #
    # during the FLASH/TOOL access.                         #
    ######################################################### 
    if ( check_for_cstate_support &&
         check_for_32_bit_non_xen_os &&
         check_for_cstate_3_enabled &&
         check_for_affected_cpu &&
         check_for_single_cpu )
    then
         taskset -c 0 $CSTATE_FIX_CMD &
    fi
    return 0
}

################################ MAIN ######################################
# Always run these

detect_os_variant
if [ "$1" == "--stop" ]
then
	check_cstatefix_running
	if [ $? == 1 ]
	then
	    disable_cstate_fix
	fi
else
	check_cstatefix_running
	if [ $? == 0 ]
	then
		enable_cstate_fix
	fi
fi
exit $?
