#!/bin/sh
# Copyright © 2015 - present Lenovo.  All rights reserved.
# Confidential and Proprietary.
# File: usblancfg.sh
# Copyright:    Lenovo Corporation
# Author:   qinzl@lenovo.com
# Version:  0.1

# Change History
# 2015/1/16 Implement the following 3 interface
#	1. echo "--bringdown MAC    bring down interface specified by MAC"
#     2. echo "--num FILE         get usb device count, and write MACs to FILE"
#     3. echo "--bringup MAC  IP    bring up interface specified by MAC, give it a default IP"
        
# List of possible return codes from this script ...
#   0   usbXX interface successfully brought up
#   1   Unable to load cdc_ether or usbnet driver load_driver()
#   2   Unable to find the CDC Ethernet interface in find_interface()
#   3   Failed to ping the IMM at address 169.254.X.118 where X is $IMM_SUBNET
#   4   One or more interfaces failed during bringup
#   5   Parsing Error
#   6   nodes present less than input nodes number
#   7   CDC Driver is not loaded
#   8   cannot find mac as usb interface
#   171 IMM system not detected
#   172 Unknown Operating System
#   173 Missing tools or drivers(lsusb,lsmod,modprobe) in Customer's env

# wangliang workaroud for spawnprocess issue: spawnprocess will remove /sbin/ from defaul list
PATH=$PATH:/sbin/
# Enable/Disable bash debug ...
set -x

# Set our core variables ...
IMM_ETHER_IPADDR=169.254.95.118
IMM_SUBNET=95
#This variable is only used when doing the final
#ping to all the interfaces
IMM_SUBNETS_USED=""

#This variable is used to keep track of which interfaces are available to be tried when using ping to determine subnet.
#IMM guarantees that we shouldn't ever get to the case where we have to search all of these subnets. I put all of them
#here just as backup. Also, the IMM guarantees that the subnets will start at 95 and work upwards by one from there.
IMM_SUBNET_STARTING_ADDR="95;96;97;98;99;100;101;102;"

CDC_ETHER_IPADDR=169.254.$IMM_SUBNET.123
CDC_ETHER_NETMASK=255.255.255.0

#Used to set up the temp address for pinging
CDC_ETHER_TEMP_IPADDR=169.254.95.119
CDC_ETHER_TEMP_NETMASK=255.255.0.0
IMM_MAX_DEVICE_COUNT=4


#This variable will be used to store the MAC addresses of multiple IMM's visible to the OS
#They will be separated with a ",".
IMM_MACADDR_STRING=""
#Keeps track of how many IMMs were found
IMM_DEVICE_COUNT=0

IBM_CDC_ETHER_USB_VID_PID="04b3:4010"
LV_CDC_ETHER_USB_VID_PID="17ef:b001"
CFC_CDC_ETHER_USB_VID_PID="046b:ffb0"
SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/sysconfig/network-scripts"
IBM_GENERATED_CONFIG_FILE_TOKEN="File created by IBM"
IBM_GENERATED_CONFIG_FILE_DATA=""
RUN_STATIC_SETUP=1
NODE_NUM=-1

IS_XEN=1 #Set to false by default
VMWARE_4=1 #Set to false by default

# Keep track of driver status before we do anything
IMM_DRIVER_STATUS=0
IMM_IFACE_STATUS=0
VMWARE_4_TEMP_FILE="/tmp/flash.ibm"

MAC_ADDR=""
INIT_IP="169.264.95.120"

IP_CMD_EXIST=0 #Set to false by default
command -v ip && IP_CMD_EXIST=1

IP_CMD_CAN_BE_USED=0 #Set to false by default
ip addr && IP_CMD_CAN_BE_USED=1 


#Parse the arguments passed to the script and set up global booleans
parse_command_line_args() {
    while [ $# -gt 0 ]
    do
        ARG=`echo $1 | tr '[:upper:]' '[:lower:]'`
        shift
        if [ $ARG == "--staticip" ]
        then
            #Can only explicitely run static setup on single node
            #Is only seen in the single node section of the main() code below
            RUN_STATIC_SETUP=0
			CDC_ETHER_IPADDR=`echo $1 | tr '[:lower:]' '[:upper:]'`
		elif [ $ARG == "--node" ]
        then
            NODE_NUM=`echo $1 | tr '[:lower:]' '[:upper:]'`
        elif [ $ARG == "--restore" ]
        then
            bring_down_all_cdc
        elif [ $ARG == "--bringdown" ]
        then
            MAC_ADDR=`echo $1 | tr '[:lower:]' '[:upper:]'`
            shift			
            bring_down_cdc_by_mac
        elif [ $ARG == "--remove" ]
        then
            unload_driver
            exit 0
        elif [ $ARG == "--status" ]
        then
            if [ "$IMM_DRIVER_STATUS" == "0" ]
            then
                exit 7
            else
                exit 0
            fi
        elif [ $ARG == "--num" ]
        then
            get_nodes_num $1
	     shift
	     break
        elif [ $ARG == "--bringup" ]
        then
            MAC_ADDR=`echo $1 | tr '[:lower:]' '[:upper:]'`
            shift
            INIT_IP=$1
            bring_up_cdc_by_mac    
        elif [ $ARG == "--nodes" ]
        then
            check_nodes_num $1
            shift
	    elif [ $ARG == "--get-vswif-mac" ]
        then
            get_vswif_mac $1 $2
            shift
        elif [ $ARG == "--is-ipv6-disabled" ]
        then
            is_ipv6_disabled $1
            shift
        elif [ $ARG == "--delip" ]
        then
            del_ip $1 $2
            shift
        elif [ $ARG == "--addip" ]
        then
            if [ ""x != $3x ];then
                add_ip $1 $2 $3
            else
                add_ip $1 $2 255.255.255.0
            fi
            shift
	    fi	
    done
}

#judge if usblan ipv6 config is enabled.
is_ipv6_disabled() 
{
    PREFIX="sysctl -b net.ipv6.conf."
    POSIX=".disable_ipv6"

    ALL_CONFIG=$(${PREFIX}all${POSIX})
    echo "${PREFIX}$1${POSIX}=${ALL_CONFIG}"
    if [ $ALL_CONFIG == "1" ]
    then
        exit 1
    fi

    DEVICE_CONFIG=$(${PREFIX}$1${POSIX})
    echo "${PREFIX}$1${POSIX}=${DEVICE_CONFIG}"
    if [ $DEVICE_CONFIG == "1" ]
    then
        exit 1
    fi

    echo "ipv6 config is enbaled."
}

del_ip() {
    if [ $# -lt 2 ];then
        echo "The option number is less then 2"
        exit 1
    fi
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1 ]; then
        ip addr del $2 dev $1
    else
        ifconfig $1 del $2
    fi
}

add_ip() {
    if [ $# -lt 2 ];then
        echo "The option number is less then 2"
        exit 1
    fi
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1 ]; then
        ip addr add $2/$3 dev $1
    else
        ifconfig $1 add $2 netmask $3
    fi

}

# Detect which Linux distribution we are running, in the case of RHEL 4 set CDC_DRIVER_NAME to "usbnet"
detect_os_variant() {
    if [ /etc/redhat-release -nt /etc/vmware-release ]
    then
        VER=`sed "s/.*release //" /etc/redhat-release | awk '{print $1}'|cut -c 1`
        MAJOR_OS=RHEL

        if [ $VER -ge 5 ]
        then
            REL=`sed "s/.*release //" /etc/redhat-release | awk '{print $1}'|cut -c 3`
            CDC_DRIVER_NAME=cdc_ether
        fi
        if [ $VER -le 4 ]
        then
            REL=`awk '{print $10}' /etc/redhat-release | cut -c 1`
            CDC_DRIVER_NAME=usbnet
        fi
        echo "RedHat Enterprise Linux Version $VER Update $REL found ..." >&1
    elif [ -f /etc/elie_version ]
    then
        MAJOR_OS=RHEL
        CDC_DRIVER_NAME=cdc_ether
    elif [ -f /etc/SuSE-release ]
    then
        VER=`awk 'NR==2{print $3}' /etc/SuSE-release`
        REL=`awk 'NR==3{print $3}' /etc/SuSE-release`
        CDC_DRIVER_NAME=cdc_ether
        MAJOR_OS=SLES
        echo "SuSE Linux Enterprise Server Version $VER Service Pack $REL found ..." >&1
        if [ $VER -ge 11 ]
        then
            #After the bridged connections are set up in XEN, they persist to non-XEN SLES 11.
            #Because of this, we need to check for a bridged connection to the cdc interface
            #before we attempt to set the usb0.
            IS_XEN=0
        fi
        SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/sysconfig/network"
    elif [ -f /etc/SUSE-brand ]
    then 
        VER=`awk 'NR==2{print $3}' /etc/SUSE-brand`
        CDC_DRIVER_NAME=cdc_ether
        MAJOR_OS=SLES
        echo "SUSE Linux Enterprise Server Version $VER found ..." >&1
        IS_XEN=0
        SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/sysconfig/network"
    elif [ -f /etc/os-release ]
    then
        MAJOR_OS=`awk -F"[\"\"]" 'NR==1{print $2}' /etc/os-release`
        if [[ $MAJOR_OS == Debian* ]]
        then
            MAJOR_OS=Debian
            echo "$MAJOR_OS Linux Server Version"
            CDC_DRIVER_NAME=cdc_ether
            SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/network"
        fi
        if [ $MAJOR_OS == "Ubuntu" ]
        then
            VER=`awk -F"[\"\"]" 'NR==2{print $2}' /etc/os-release|awk -F"." '{print $1}'`
            echo "$MAJOR_OS Linux Server Version $VER found ..." >&1
            CDC_DRIVER_NAME=cdc_ether
            SYSCONFIG_NETWORK_SCRIPTS_DIR="/etc/network"
        fi
    elif [ `vmware -v | awk '{print $1}'` == "VMware" ]
    then
        #Since vmware -v output has changed between releases need to detect the release differently
        VER3=`vmware -v | awk '{print $4}' | cut -c 1`
        VER4=`vmware -v | awk '{print $3}' | cut -c 1`
        if [ $VER3 == 3 ]
        then
            VER=3
        elif [ $VER4 == 4 ]
        then
            VER=4

            #print interface config info here for debug
            esxcfg-vswitch -l
            esxcfg-vswif -l
        fi

        # Set MAJOR_OS
        MAJOR_OS=VMWARE

        # Load correct driver
        if [ $VER == 3 ]
        then
            #Driver does not exist in VMware ESX Server 3.5 prior to U4
            CDC_DRIVER_NAME=CDCEther
        fi
        if [ $VER == 4 ]
        then
            CDC_DRIVER_NAME=cdc_ether
            VMWARE_4=0
        fi
        echo "VMware ESX Server $VER found ..." >&1
    else
        echo "Unable to detect operating system, exiting." >&1
        exit 172
    fi

    #print interface config info here for debug
    ip_command
    route_command
}

#get nodes number in system,the accual nodes number should be (exitCode-200)
get_nodes_num() {
    macFile=$1
    exitCode=`expr $IMM_DEVICE_COUNT + 200`
    if [ ! "$macFile" == "" ]
    then
        mkdir -p ` echo $macFile | sed "s/\/[^\/]*$//" ` 1>/dev/null 2>&1
        echo $IMM_MACADDR_STRING > $macFile        
    fi
	
	exit $exitCode
}

#if nodes present less than $1,then exit 6
check_nodes_num() {
    nodesNum=$1
    #check validity of param
    if [ "$nodesNum" == "" ]
    then
        echo "no nodes number followed."
        exit 5
    fi

    numCnt=`expr $nodesNum : "[0-9]*$"`
    if [ $numCnt == 0 ]
    then
        echo "nodes number is invalid."
        exit 5
    fi

    #check if all nodes present
    if [ $IMM_DEVICE_COUNT -lt $nodesNum ]
    then
        echo "Not all nodes present."
        exit 6
    fi
}



#check if ping to the right imm, using USB_SUBNET and INTERFACE_IP_LIST
check_imm_route() {
    correct_route=1
    #find the real route
    first_route=`route_command | grep 169.254.$USB_SUBNET | head -1 | awk '{print $8}' `
    if [ -z $first_route ]
    then
        first_route=`route_command | grep 169.254.0 |head -1 | awk '{print $8}' `
    fi

    #check if first_route is in the INTERFACE_IP_LIST
    TEMP_IFS5=$IFS
    IFS=$(echo -en ",")
    for route in $INTERFACE_IP_LIST
    do
    if [ "$first_route" == "$route" ]
    then
        correct_route=0
        break
    fi
    done
    IFS=$TEMP_IFS5

    return $correct_route
}
#get all the mac addr of usb dev,this func should be called after detect_os_variant and before all other func.
get_mac_addr() {
    if [ $MAJOR_OS == VMWARE ]
    then
        if [ $VER == "3" ]
        then
            #By deleting the driver and re-installing it, we insure that the functional MAC address of the
            #interface is the last loaded address and we don't have a bogus MAC like 00:00:00:00
            unload_driver
            load_driver
            vmware_check_for_imm_get_mac
            if [ $IMM_DRIVER_STATUS == 0 ]
            then
                unload_driver
            fi
        elif [ $VER == 4 ]
        then
            # No need for load_driver on VMWare 4, guaranteed to be loaded
            vmware_check_for_imm_get_mac
        fi
    else
        load_driver
        check_for_imm_get_mac
        if [ $IMM_DRIVER_STATUS == 0 ]
        then
            unload_driver
        fi
    fi
}
#find all the usb interface where the ip should be configured
#and put them in INTERFACE_IP_LIST
find_all_ip_interface() {
    INTERFACE_IP_LIST=""
    FIND_GOOD=0

    TEMP_IFS1=$IFS
    IFS=$(echo -en ",")
	
    echo "IMM_MACADDR_STRING = "$IMM_MACADDR_STRING

    for mac in  $IMM_MACADDR_STRING
    do
        #find interface
        INTERFACE=""
        find_interface_by_mac $mac
        if [ "$INTERFACE" != "" ]
        then
            echo "no interface found for $mac"
            FIND_GOOD=1
            break
        fi

        #convert interface to INTERFACE_IP
        if [ $MAJOR_OS == VMWARE ] && [ $VER != "3" ]
        then
            vmware4_find_ip_interface $INTERFACE
        elif [ $MAJOR_OS == SLES ]
        then
            find_bridged_connection $mac
            if [ $BR_INTERFACE_FOUND == 0 ]
            then
                INTERFACE_IP=$BR_INTERFACE
            else
                INTERFACE_IP=$INTERFACE
            fi
        else
            INTERFACE_IP=$INTERFACE
        fi
        if [ -z "$INTERFACE_IP" ]
        then
            echo "no INTERFACE_IP found for $INTERFACE"
            FIND_GOOD=1
            break
        fi

        #add to INTERFACE_IP_LIST
        TEMP_IFS3=$IFS
        IFS=$(echo -en ",")
        for inter in $INTERFACE_IP
        do
            #filter the repeated one
            if check_repeated_interface $inter
            then
                continue
            fi
            #add to INTERFACE_IP_LIST
            INTERFACE_IP_LIST=$INTERFACE_IP_LIST$inter","
        done
    IFS=$TEMP_IFS3
    done

    IFS=$TEMP_IFS1

    return $FIND_GOOD
}

#check if it is a repeated interface in INTERFACE_IP_LIST
check_repeated_interface() {
    repeat=1
    TEMP_IFS4=$IFS
    IFS=$(echo -en ",")
    for i in $INTERFACE_IP_LIST
    do
        if [ "$i" == "$1" ]
        then
            repeat=0
            break
        fi
    done
    IFS=$TEMP_IFS4

    return $repeat
}

#find real interface by $INTERFACE
vmware4_find_ip_interface() {
    INTERFACE_IP=""
    TEMP_IFS2=$IFS
    IFS=$(echo -en "\n\b")

    #find portgroup by interface,there could be mutiple PG for one interface
    PGs=`esxcfg-vswitch -l | grep $1 | sed -n '2,$p' | sed 's/[0-9]\{1,\}[ ]\{1,\}[0-9]\{1,\}[ ]\{1,\}.*'$1'.*[ ]\{0,\}$//' | sed 's/[ \t]*$//;s/^[ \t]*//' `

    for PG in $PGs
    do
        #find ip interface by portgroup
        interface_temp=`esxcfg-vswif -l | grep $PG | awk '{print $1}'`

        #add ip interface to INTERFACE_IP
        for inter in $interface_temp
        do
           INTERFACE_IP=$INTERFACE_IP$inter","
        done
    done

    IFS=$TEMP_IFS2

}

#this is a common interface to find interface name by mac addr
#for vmware 4 the interface is vusb which is not ip configured, the link interface is vswif where you can find ip configuration
find_interface_by_mac() {
    iMac=$1
    ret=0
    if [ $MAJOR_OS == VMWARE ] && [ $VER == "3" ]
    then
        vmware_3_find_interface $iMac
    else
        find_interface $iMac
    fi
}

find_mac_by_interface() {
    CUR_INTERFACE=$1
    CURRENT_PARSED_MAC=`cat /sys/class/net/$CUR_INTERFACE/address`
    if [ ! -z "$CURRENT_PARSED_MAC" ]
    then
        MAC_FIND=`echo $CURRENT_PARSED_MAC | sed 's/://g' | tr "[:lower:]" "[:upper:]"`
    else
        echo "ERROR:no /sys/class/net/$CUR_INTERFACE/address found!"
    fi	
}

get_vswif_mac(){
    MAC_FROM_CLI=$1
    macFile=$2
    MAC_TO_WRITE=$MAC_FROM_CLI

    #for vmware, we need to find the vswif mac, other os, we do nothing    
    if [ $MAJOR_OS == VMWARE ]
    then
        $INTERFACE=""
        $MAC_FIND=""
        find_interface_by_mac $MAC_FROM_CLI
        if [ $INTERFACE != "" ]
        then
            # First find the PortGroup Name associated with the physical CDC nic
            PORTGROUP=`esxcfg-vswitch -l | grep $INTERFACE | tail -1 | awk '{print $1}'`
            if [ ! -z "$PORTGROUP" ]
            then        				
            	#echo "The Portgroup is " $PORTGROUP
                # Next get the vswif associated with this PortGroup
                NEW_INTERFACE=`esxcfg-vswif -l | grep $PORTGROUP | head -1 | awk '{print $1}'`
                if [ ! -z "$NEW_INTERFACE" ]
                then
                    echo "The Interface is " $NEW_INTERFACE
                    find_mac_by_interface $NEW_INTERFACE
                    if [ $MAC_FIND != "" ]
                    then
                        echo "find vswif mac " $MAC_FIND 
                        MAC_TO_WRITE=$MAC_FIND
                    fi
                fi
            fi		
        fi
    fi
    		
    if [ ! "$macFile" == "" ]
    then
        mkdir -p ` echo $macFile | sed "s/\/[^\/]*$//" ` 1>/dev/null 2>&1
        echo $MAC_TO_WRITE > $macFile        
    fi
	
    exit 0		
}

# Chech the current status of the driver
get_driver_status() {
    if [ $MAJOR_OS == VMWARE ] && [ $VER == "4" ]
    then
        CDC_DRIVER=`esxcfg-module -q  | grep cdc_ether`
        if [ ! -z "CDC_DRIVER" ]
        then
            IMM_DRIVER_STATUS=1
        fi
        return
    fi

    CDC_DRIVER_LOADED=`lsmod | awk '{ print $1 }' | grep $CDC_DRIVER_NAME`
    
    if [ "$CDC_DRIVER_LOADED" == "$CDC_DRIVER_NAME" ]
    then
        IMM_DRIVER_STATUS=1
    fi
}

# Load the cdc_ether or usbnet driver, depending on the OS ...
load_driver() {

    #vmware 4 or later, the driver guaranteed to be loaded, so just check if it is enable
    if [ $MAJOR_OS == VMWARE ] && [ $VER -ge 4 ]
    then
        esxcfg-module -e cdc_ether.o
        if [ "$?" != "0" ]
        then
            echo "fail to enable cdc module for vmware 4. Please check if load cdc module when boot."
            exit 173
        fi
						
    return
    fi

    CMD_STATUS=0
    CDC_DRIVER_LOADED=`lsmod | awk '{ print $1 }' | grep $CDC_DRIVER_NAME`
    #check if lsmod succeed on user's env
    if [ "$?" != "0" ]
    then
        CMD_STATUS=1
    fi
    
    if [ "$CDC_DRIVER_LOADED" != "$CDC_DRIVER_NAME" ]
    then
        if !( modprobe $CDC_DRIVER_NAME)
        then
            echo "Could not load driver, $CDC_DRIVER_NAME ..." >&2
            if [ $CMD_STATUS == 1 ]
            then
                #maybe user's env has already loaded the CDC_DRIVER, and we are not able to detect the status bcz of lsmod fail,
                #exit 173 so that invoker aware of this
                exit 173
            else
            	exit 1
            fi
        fi
    fi
}

#saw a problem in VMWare 3.5 where the driver was already loaded and we had problems finding the MAC address
unload_driver(){
    #vmware 4 , the driver guaranteed to be loaded, so just disable it
    if [ $MAJOR_OS == VMWARE ] && [ $VER == "4" ]
    then
        esxcfg-module -d cdc_ether.o
        return
    fi
		
    CDC_DRIVER_LOADED=`lsmod | awk '{ print $1 }' | grep $CDC_DRIVER_NAME`
    if [ "$CDC_DRIVER_LOADED" == "$CDC_DRIVER_NAME" ]
    then
        if !( modprobe -r $CDC_DRIVER_NAME)
        then
            echo "Could not delete driver, $CDC_DRIVER_NAME ..." >&2
        fi
    fi
}

# Determine if we are on an IMM system and for Linux get the MAC here
check_for_imm_get_mac() {
    #NOTE: This function will not be able to find the MAC address for VMWare 3. The address will be discovered in vmware_3_find_interface()

    # RHEL 4 has a bug with "lsusb -v -d VID:PID", but -s works with the -v, so we pull the
    # BUS:DEVICE from our device line and use "lsusb -v -s BID:DID" which does work on RHEL 4
    echo "Finding MAC address...."
    #add these two lines in case it is called repeated
    IMM_MACADDR_STRING=""
    IMM_DEVICE_COUNT=0
    # Get the output from lsusb
    LSUSB_OUT=`lsusb`
    #check if lsusb succeed on user's env
    if [ "$?" != "0" ]
    then
        echo "lsusb failed,not able to go on running cdc_interface.sh."
        exit 173
    fi

    # Set bash internal field separator to newline
    ORIG_IFS=$IFS
    IFS=$(echo -en "\n\b")

    # Parse through output of lsusb, looking for our known USB VID:PID
    LSUSB_BUS=null

    for i in $LSUSB_OUT
    do
        # Parse out USB VendorID:ProductID
        LSUSB_OUT_VID_PID=`echo $i | awk '{print $6}'`

        # Check against our known USB VendorID:ProductID
        if [ "$LSUSB_OUT_VID_PID" == "$IBM_CDC_ETHER_USB_VID_PID"  ] || [ "$LSUSB_OUT_VID_PID" == "$LV_CDC_ETHER_USB_VID_PID" ] || [ "$LSUSB_OUT_VID_PID" == "$CFC_CDC_ETHER_USB_VID_PID" ]
        then
            # Parse out our BusID:DeviceID
            # We remove the zero because RHEL 5.4 doesn't like it.
            LSUSB_BUS=`echo $i | awk '{print $2}' | sed 's/^0*//g'`
            LSUSB_DEVICE=`echo $i | awk '{print $4}' | sed 's/://g' | sed 's/^0*//g'`

            LSUSB_iMAC=`lsusb -v -s $LSUSB_BUS:$LSUSB_DEVICE | grep -i mac | head -1 | awk '{print $3}' | tr "[:lower:]" "[:upper:]"`

            if [ -z "$LSUSB_iMAC" ]
            then
                #String is empty we have a problem here.
                echo "The MAC address in lsusb does not present itself."
            else
                IMM_MACADDR_STRING=$IMM_MACADDR_STRING$LSUSB_iMAC","
                IMM_DEVICE_COUNT=`expr $IMM_DEVICE_COUNT + 1`
                echo "Found " $IMM_DEVICE_COUNT " Devices "
            fi

        fi
    done

    # Set bash internal field separator back to original value
    IFS=$ORIG_IFS

    if [ $IMM_DEVICE_COUNT == 0  ]
    then
        echo "IMM system not detected"
        echo "Show all USB devices via lsusb:"
        lsusb
        exit 171
    fi
}


# Parse the kernel ring buffer for cdc ethernet driver instances
# and compare with our known MAC to find the interface name
find_interface() {

    LSUSB_iMAC=$1
    echo "Looking for the interface name..."
    INTERFACE_FOUND=0
    INTERFACE=""

    # Get the USB interface list all usb network instances.  We look at /sys/class/net/* for usbX device
    # names, this would ensure the device is in use and registered.
    #SYS_CDC_ETHER_LIST=`ls /sys/class/net | grep "usb"`
    SYS_CDC_ETHER_LIST=`ls /sys/class/net | grep -E '(usb|eth|en)'`

    # Set bash internal field separator to newline
    ORIG_IFS=$IFS
    IFS=$(echo -en "\n\b ")

    echo "Looking for device with MAC of " $LSUSB_iMAC
    for i in $SYS_CDC_ETHER_LIST
    do
        INTERFACE_TEST=`echo $i`

        # Get current MAC
        CURRENT_PARSED_MAC=`cat /sys/class/net/$INTERFACE_TEST/address`
        COLON_STRIPPED_CURRENT_PARSED_MAC=`echo $CURRENT_PARSED_MAC | sed 's/://g' | tr "[:lower:]" "[:upper:]"`

        if [ "$COLON_STRIPPED_CURRENT_PARSED_MAC" == "$LSUSB_iMAC" ]
        then
            INTERFACE=$INTERFACE_TEST

            # Set INTERFACE_FOUND
            INTERFACE_FOUND=1

            # Set INTERFACE_MAC
            INTERFACE_MAC=$CURRENT_PARSED_MAC

            break
        fi
    done

    IFS=$(echo -en "\n\b")
    if [ "$INTERFACE_FOUND" == 0 ]
    then
        # Try getting the mac address out of 'dmesg' instead
        # Get the MAC for usb device 04b3:4010 and all MACs from all cdc_ether driver instances
        DMESG_CDC_ETHER_MACS=`dmesg | grep "CDC Ethernet Device"`

        # Parse through cdc_ether driver instance MACs, match up with MAC from lsusb of IBM's CDC device
        for i in $DMESG_CDC_ETHER_MACS
        do
            # Only parse lines that are for driver registration ...
            if [ `echo $i | awk '{print $2}'` == "register" ]
            then
                # Parse out the MAC from this driver registration instance
                CURRENT_PARSED_MAC=`echo $i | awk 'NR==1{print $9}'`
                COLON_STRIPPED_CURRENT_PARSED_MAC=`echo $CURRENT_PARSED_MAC | sed 's/://g' | tr "[:lower:]" "[:upper:]"`
                # Test parsed MAC against our known MAC
                if [ "$COLON_STRIPPED_CURRENT_PARSED_MAC" == "$LSUSB_iMAC" ]
                then
                    # Found our device, set preliminary insterface name
                    INTERFACE_PRELIM=`echo $i | awk 'NR==1{print $1}'`

                    # Parse out the trailing colon from the interface name
                    INTERFACE=`echo -n "$INTERFACE_PRELIM" | sed 's/://g' | sed 's/ //g'`

                    # Set INTERFACE_FOUND
                    INTERFACE_FOUND=1
                    INTERFACE_MAC=$CURRENT_PARSED_MAC

                    # Don't break here, instead continue the for loop so that
                    # we make sure we get the most recent instance of the driver loading
                    # that also matches our known MAC, in case the interface name was different
                    # for a previous registration of the driver for our MAC
               fi
           fi
       done
    fi

    # Set bash internal field separator back to original value
    IFS=$ORIG_IFS


    # VMware 4 hack, until we figure out what replace lsusb
    #if [ $MAJOR_OS == "VMWARE" ] && [ $VER == "4" ]
    #then
    #   INTERFACE_FOUND=1
    #   INTERFACE=usb0
    #fi

    # If interface was not found, exit 2
    if [ "$INTERFACE_FOUND" == "0" ]
    then
        echo "Could not locate the CDC Interface"
        exit 2
    fi
}

#Compare the colon stripped mac address of the usb interface to those of the brX interfaces to find the correct bridge
#This is needed only for SLES 11 with XEN and potential newer versions of SLES 11
find_bridged_connection() {
    LSUSB_iMAC=$1
    BR_INTERFACE_FOUND=1
    echo "Looking for interface bridge..."
    # Get the br interface list all br network instances.  We look at /sys/class/net/* for brX device
    # names, this would ensure the device is in use and registered.
    SYS_BR_LIST=`ls /sys/class/net | grep "br"`

    # Set bash internal field separator to newline
    ORIG_IFS=$IFS
    IFS=$(echo -en "\n\b")

    for i in $SYS_BR_LIST
    do
        BR_INTERFACE_TEST=`echo $i`

        # Get current MAC
        BR_CURRENT_PARSED_MAC=`cat /sys/class/net/$BR_INTERFACE_TEST/address`
        BR_COLON_STRIPPED_CURRENT_PARSED_MAC=`echo $BR_CURRENT_PARSED_MAC | sed 's/://g' | tr "[:lower:]" "[:upper:]"`

        if [ "$BR_COLON_STRIPPED_CURRENT_PARSED_MAC" == "$LSUSB_iMAC" ]
        then
            BR_INTERFACE=$BR_INTERFACE_TEST

            # Set INTERFACE_FOUND
            BR_INTERFACE_FOUND=0

            # Set INTERFACE_MAC
            BR_INTERFACE_MAC=$BR_CURRENT_PARSED_MAC

            break
        fi
    done

    if [ "$BR_INTERFACE_FOUND" == 1 ]
    then
        #We did not find a bridged connection. This means that the user is either
        #running standard SLES 11 w/o XEN, or the user has not configured their
        #hypervisor to set up the briged network connections. Now we need to attempt
        #to set up the usb0 instead of brX.
        echo "Could not locate the Bridge for the CDC interface."
        echo "Attempting to bring up alternate interface..."
        IS_XEN=1

    fi

    # Set bash internal field separator back to original value
    IFS=$ORIG_IFS

}
#VMWare3 function to find the interface
vmware_3_find_interface() {
    TEMP_MAC=$1
    INTERFACE=""
    echo "TEMP_MAC = " $TEMP_MAC
    INTERFACE=`dmesg | sed 's/://g' | tr "[:upper:]" "[:lower:]" | grep -i $TEMP_MAC | tail -1 | awk '{print $2}'`
    echo "The interface is " $INTERFACE
}

# VMware function to find the mac address
vmware_check_for_imm_get_mac() {
    echo "Getting the mac address of the network device."
    #VMWare4 has a bug where lsusb -v -s doesn't work correctly, and since RHEL4
    #has a bug with lsusb -v -d we needed a separate function
    IMM_MACADDR_STRING=""
    IMM_DEVICE_COUNT=0
    # Set bash internal field separator to newline and space
    ORIG_IFS=$IFS
    IFS=$(echo -en "\n\b ")

    if [ $VMWARE_4 == 1 ] #not VMWare4
    then
        #VMWare 3
        DMESG_OUT=`dmesg | grep -i rndis | grep -i cdc | grep -i ibm | awk '{print $2}' | head -1`

        TEMP_MAC=`dmesg | grep $DMESG_OUT | egrep "[[:alnum:]][[:alnum:]]:[[:alnum:]][[:alnum:]]" | tail -1 | awk '{print $3}' | tr "[:upper:]" "[:lower:]" | sed 's/://g' | sed 's/ //g'`
        IMM_MACADDR_STRING=$IMM_MACADDR_STRING$TEMP_MAC","
        IMM_DEVICE_COUNT=`expr $IMM_DEVICE_COUNT + 1`

        echo "MAC_ADDR_STRING = " $IMM_MACADDR_STRING

    else
        #VMWARE4

        echo "Finding MAC address...."

        # Get the output from lsusb
        LSUSB_OUT=`lsusb -v -d $IBM_CDC_ETHER_USB_VID_PID | grep -i mac | awk '{print $3}'`

        # Parse through output of lsusb, looking for our known USB VID:PID
        LSUSB_BUS=null

        for i in $LSUSB_OUT
        do
             IMM_MACADDR_STRING=$IMM_MACADDR_STRING$i","
             IMM_DEVICE_COUNT=`expr $IMM_DEVICE_COUNT + 1`
        done
		
	    # Get the output from lsusb
        LSUSB_OUT1=`lsusb -v -d $LV_CDC_ETHER_USB_VID_PID | grep -i mac | awk '{print $3}'`

        for i in $LSUSB_OUT1
        do
             IMM_MACADDR_STRING=$IMM_MACADDR_STRING$i","
             IMM_DEVICE_COUNT=`expr $IMM_DEVICE_COUNT + 1`
        done

        LSUSB_OUT2=`lsusb -v -d $CFC_CDC_ETHER_USB_VID_PID | grep -i mac | head -1 | awk '{print $3}'`

        for i in $LSUSB_OUT2
        do
             IMM_MACADDR_STRING=$IMM_MACADDR_STRING$i","
             IMM_DEVICE_COUNT=`expr $IMM_DEVICE_COUNT + 1`
        done
        # In Vmware4, dmesg can not offer mac address, so we could do nothing if lsusb fail here
    fi

    # Set bash internal field separator back to original value
    IFS=$ORIG_IFS

    # If interface was not found, exit 2
    if [ $IMM_DEVICE_COUNT -lt 1 ]
    then
        echo "IMM System not detected."
        echo "Show all USB devices via lsusb:"
        lsusb
        exit 171
    fi
}



#Determine the correct interface to use to display information
get_correct_interface() {
    echo "Determining interface name"
    if [ $IS_XEN == 0 ]
    then
        CORRECT_INTERFACE=$BR_INTERFACE
    else
        CORRECT_INTERFACE=$INTERFACE
    fi

}


#bring up cdc by interface
bring_up_cdc() {
    echo "Attempting to bring up the interface"
    get_correct_interface

    #Make sure that in the XEN environment the usb0 interface is down
    #Technically, the bridge on the XEN environment needs usb0 to be up, but not configured to run correctly.
    #This is the reason usb0 is taken down before the bridge connection, since the bridge connection will bring it
    #back up correctly.
    if [ $IS_XEN == 0 ]
    then
        #Note:The configuration file for this interface has already been deleted in write_config_file()
        ip_command $INTERFACE down
    fi
    
    # all the os need down the interface then up it.
    #ip_command $INTERFACE down

    # for rhel6, Pausing for 1 seconds to let system quiesce
    if [ $MAJOR_OS == RHEL ] && [ $VER == 6 ]
    then
        echo "Pausing for 1 seconds to let system quiesce..."
        sleep 1
    fi
	
    # Try to bring up the interface ...
    if [ -z $INIT_IP ]
    then
	ip_command $CORRECT_INTERFACE up
    else
	ip_command $CORRECT_INTERFACE $INIT_IP
    fi
    
    if [ $? != 0 ]
    then
        echo "We could not bring up interface $CORRECT_INTERFACE" >&2
        exit 1
    fi
    echo "Brought up the interface successfully"
    return 0
}

#bring down the cdc interfaces match the MAC_ADDR
#This function is only envoked from the argument --bringdown to this script
# For the interface, find it and bring it down
# then unload the driver
# then  restore its original driver state
# this is needed because some OSes will bring up all interfaces when the driver is loaded

do_bring_down_cdc_by_mac(){
    IMM_DRIVER_HANDLED=0
	
    if [ $MAJOR_OS == VMWARE ]
    then
        # Set bash internal field separator to comma
        TEMP_IFS=$IFS
        IFS=$(echo -en ",")

        if [ $VER == "3" ]
        then
            #####
            ##
            ## VMWARE 3 is not supported by multi-node systems
            ##
            #####

            #####
            #We keep the for loop here because vmware 3 uses dmesg to find the interface, and because of this
            #it is possible to get multiple instances of the same interface back.
            #####

            vmware_3_find_interface $MAC_ADDR
            echo  $INTERFACE "found,bringdown it and remove config file."
            ifdown $INTERFACE

            #careless of original interface status,just restore original config file
            #restore original driver status
            IMM_DRIVER_STATUS=0
            DRV=`grep "IMM_DRIVER_STATUS" $SYSCONFIG_NETWORK_SCRIPTS_DIR/ifcfg-$INTERFACE | awk -F= '{ print $2 }'`
            if [ "$DRV" == "0" ]
            then
                IMM_DRIVER_STATUS=0
            elif [ "$DRV" == "1" ]
            then
                IMM_DRIVER_STATUS=1
            else
                IMM_DRIVER_HANDLED=0
                IMM_DRIVER_STATUS=1
            fi
            
            restore_config_files ifcfg-$INTERFACE
			
            if [ "$IMM_DRIVER_STATUS" == "1" ]
            then
                if [ "$IMM_DRIVER_HANDLED" == "0" ]
                then
                    load_driver
                    IMM_DRIVER_HANDLED=1
                    #give the driver a chance to load
                    sleep 10
                fi
            fi
        elif [ $VER == 4 ]
        then
            # No need for write_config_file on VMWare 4, esxcfg-vswitch and esxcfg-vswif write their own configs
            # No need for disable_zeroconf on VMWare 4, esxcfg-vswif takes care of the unneeded link-local routes

            #If no temp file, then we did nothing so leave as is
			
			#modify by cindy, we bring down anyway when called to
            #if [ -f $VMWARE_4_TEMP_FILE ]
            #then
                if [ $IMM_DEVICE_COUNT -gt 0 ]
                then
                    find_interface $MAC_ADDR
                    vmware4_delete_vswif_by_usb $INTERFACE
                    			
                    # delete vswitch
                    present=`esxcfg-vswitch -l|grep IBM_CDC_vSwitch0|awk '{print $1}'`
                    if [ ! -z $present ]
                    then
                        esxcfg-vswitch -d IBM_CDC_vSwitch0
                    fi
                fi
				
				#restore driver status when it is recorded,else, do nothing
				if [ -f $VMWARE_4_TEMP_FILE ]
				then
                    DRV=`grep "IMM_DRIVER_STATUS" $VMWARE_4_TEMP_FILE | awk -F= '{ print $2 }'`
                    if [ "$DRV" == "0" ]
                    then
                        unload_driver
                    fi
				fi

            #    rm -f $VMWARE_4_TEMP_FILE
            #fi #Temp File
        fi # VMWare 4

        IFS=$TEMP_IFS
    else  #Linux
        # Set bash internal field separator to comma
        TEMP_IFS=$IFS
        IFS=$(echo -en ",")

        if [ $IMM_DEVICE_COUNT -gt 0 ]
        then
            find_interface $MAC_ADDR
            if [ $IS_XEN == 0 ]
            then
                #We attempt to find the bridged interface after the normal usb interface so that the functions following
                #this one can use the IS_XEN value that is set in find_bridged_connection to determine whether to use
                #$INTERFACE or $BR_INTERFACE when writing the file.
                find_bridged_connection $MAC_ADDR
            fi

            if [ $IS_XEN == 0 ]
            then
                INTERFACE=$BR_INTERFACE
            fi

            echo  $INTERFACE "found,bringdown it."
	    ip_command $INTERFACE down

        fi
        IFS=$TEMP_IFS
    fi

    return 0
}
#bring down the cdc interfaces match the MAC_ADDR
#This function is only envoked from the argument --bringdown to this script
# For the interface, find it and bring it down
# then unload the driver
# then  restore its original driver state
# this is needed because some OSes will bring up all interfaces when the driver is loaded
bring_down_cdc_by_mac(){
    IMM_DRIVER_HANDLED=0
	
    #check if MAC specifyed is one of our usb interface
    check_if_usbmac_exist $MAC_ADDR
    if [ $? != 0 ]
    then
        echo "cannot find mac as usb interface." >&1
    	exit 8
    fi
	
	do_bring_down_cdc_by_mac
	
	ret=$?
	if [ $ret != 0 ]
    then
        echo "fail to bring down cdc "$MAC_ADDR >&1
    	exit $ret
    fi

    exit 0
}

#bring down all cdc interface
bring_down_all_cdc()
{
	ret=0
    TEMP_IFS=$IFS
    IFS=$(echo -en ",")
	
    echo "begin to bring down all cdc..."
	
	
    for MAC in  $IMM_MACADDR_STRING
    do
        MAC_ADDR=$MAC
        do_bring_down_cdc_by_mac
        ret=$?
        if [ $ret != 0 ]
		then
            echo "fail to bring down "$MAC
        fi
    done

    IFS=$TEMP_IFS
    exit $ret
}


#Look and see if there is an interface that is disabled for the cdc device
vmware4_delete_previous_interfaces() {
    echo "Looking for current interface(s)..."
    #check to see if there is already vswif NIC setup, but disabled
    DISABLED_INTERFACE=`esxcfg-vswif -l | grep IBM_CDC_PG | awk '{print $1}'`

    if [ -z "$DISABLED_INTERFACE" ]
    then
        echo "Interface not disabled."
    else
        echo "Found Previous Interface(s) " $DISABLED_INTERFACE
        # Set bash internal field separator to newline
        ORIG_IFS=$IFS
        IFS=$(echo -en "\n\b ")
        for i in $DISABLED_INTERFACE
        do
            esxcfg-vswif -d $i
            if [ $? == 0 ]
            then
                echo "Deleted interface " $i
            fi
        done
        IFS=$ORIG_IFS
        return 0
    fi

    return 1
}

#delete vswif by the usb interface name
vmware4_delete_vswif_by_usb() {
    USB_INTERFACE=$1
    ret=0
    # Set bash internal field separator to newline
    ORIG_IFS=$IFS
    IFS=$(echo -en "\n\b")

    CDC_PGs=`esxcfg-vswitch -l|grep $USB_INTERFACE|sed -n '2,$p'|sed 's/[0-9]\{1,\}[ ]\{1,\}[0-9]\{1,\}[ ]\{1,\}.*'$1'.*[ ]\{0,\}$//' |sed 's/[ \t]*$//;s/^[ \t]*//'  `

    for CDC_PG in $CDC_PGs
    do
        #check to see if there is already vswif NIC setup, but disabled
        DISABLED_INTERFACE=`esxcfg-vswif -l | grep $CDC_PG | awk '{print $1}'`

        if [ -z "$DISABLED_INTERFACE" ]
        then
            echo "Interface not found."
        else
            echo "Found Interface(s) " $DISABLED_INTERFACE "using usb"

            for i in $DISABLED_INTERFACE
            do
                iusb=`esxcfg-vswif -l |grep $i|grep 169.254|awk '{print $1}'`
                if [ "$iusb" != "" ]
                then
                    esxcfg-vswif -d $i
                    if [ $? == 0 ]
                    then
                        echo "Deleted interface " $i
                    fi
                fi
            done
            ret=0
        fi
    done

    IFS=$ORIG_IFS
    ret=1
    return $ret
}

# For VMWare 4.0, bring up the interface
vmware_4_bring_up_cdc() {
    runDHCP=$1
    present=""

    # Add vswitch "IBM_CDC_vSwitch0"
    present=`esxcfg-vswitch -l|grep IBM_CDC_vSwitch0|awk '{print $1}'`
    if [ -z $present ]
    then
        esxcfg-vswitch -a IBM_CDC_vSwitch0
    fi

    # Add portgroup "IBM_CDC_PG" to vswitch "IBM_CDC_vSwitch0"
    present=`esxcfg-vswitch -l|grep IBM_CDC_PG|awk '{print $1}'`
    if [ -z $present ]
    then
        esxcfg-vswitch -A IBM_CDC_PG IBM_CDC_vSwitch0
    fi

    # Delete physical nic (pnic) $INTERFACE from any vswitch
    usb_vswitch=`esxcfg-vswitch -l|grep $INTERFACE|head -1|awk '{print $1}'`
    if [ ! -z $usb_vswitch ]
    then
        vswitch_len=`expr length "$usb_vswitch"`
        if [ "$vswitch_len" -gt "16" ]
        then
            vswitch_len=`expr $vswitch_len - 2`
            usb_vswitch=`echo $usb_vswitch|cut -c1-$vswitch_len`
        fi
        esxcfg-vswitch -U $INTERFACE $usb_vswitch

    fi

    # Add physical nic (pnic) $INTERFACE to vswitch "IBM_CDC_vSwitch0"
    esxcfg-vswitch -L $INTERFACE IBM_CDC_vSwitch0

    # Add vswif, check for existing interfaces and grab the next available one
    COUNT=0
    for (( ; ; ))
    do
        VSWIF_INT=vswif$COUNT

        if [ `esxcfg-vswif -c $VSWIF_INT` == 0 ]
        then
            echo "#IMM_IFACE_STATUS=$MAC_ADDR=$IMM_IFACE_STATUS" >> $VMWARE_4_TEMP_FILE

            if [ -z "$runDHCP" ]
            then
                echo "Setting up connection with Static Address"
                #delete the repeated vswif
                present=`esxcfg-vswif -l|grep 169.254.95.119|awk '{print $1}'`
                if [ "$present" != "" ]
                then
                    esxcfg-vswif -d $present
                fi
                # we only need one vswif with 169.254.95.119/16, then both 169.254.95.118 and 169.254.96.118 can be pinged.
                esxcfg-vswif -a -i $CDC_ETHER_TEMP_IPADDR -n $CDC_ETHER_TEMP_NETMASK -p IBM_CDC_PG $VSWIF_INT
                if [ $? == 1 ]
                then
                    esxcfg-vswif -i $CDC_ETHER_IPADDR -n $CDC_ETHER_TEMP_NETMASK $VSWIF_INT
                fi

            else
                echo "Setting up connection with DHCP address"
                esxcfg-vswif -a -i DHCP -p IBM_CDC_PG $VSWIF_INT
                if [ $? == 1 ]
                then
                    return 1
                fi
            fi
            INTERFACE=$VSWIF_INT
            break
        fi

        COUNT=$(($COUNT+1))
    done
    return 0
}
#take all of the config files and rewrite them for DHCP
vmware_4_modify_interface(){

    vmware_4_imm_already_up_get_vswif_interface

    echo "Modifying the previous interface"
    esxcfg-vswif -i DHCP $INTERFACE

}

# If VMWare 4 and if IMM is already able to be pinged, we still need to get our vswif interface name to pass back
vmware_4_imm_already_up_get_vswif_interface() {

    # First find the PortGroup Name associated with the physical CDC nic
    PORTGROUP=`esxcfg-vswitch -l | grep vusb0 | tail -1 | awk '{print $1}'`

    if [ -z "$PORTGROUP" ]
    then
        return 1
    fi
    #echo "The Portgroup is " $PORTGROUP
    # Next get the vswif associated with this PortGroup
    INTERFACE=`esxcfg-vswif -l | grep $PORTGROUP | head -1 | awk '{print $1}'`
    #echo "The Interface is " $INTERFACE

    return 0
}

# Remove all link local routes that might exist, except for our interface
# do nothing since IMM Ip is not 169.254 now
delete_link_local_routes() {
    # This link-local route exists on some distributions on interfaces besides the IMM and will prevent us from
    # talking with the IMM so we attempt to remove it here. If the route does not exist this will fail but we don't care ...
    return 0
}


# Echo the interface information to stdout
echo_interface_info() {
    get_correct_interface
    echo "INTERFACE INFORMATION:"
    IPADDR=GET_IPADDR $INTERFACE
    NETMASK=GET_NETMASK $INTERFACE 

    #Make sure this works with other OSs....
    MAC=GET_MAC $INTERFACE

    echo "INTERFACE="$CORRECT_INTERFACE >&1
    echo "IPADDR="$IPADDR >&1
    echo "NETMASK="$NETMASK >&1
    echo "MAC="$MAC >&1
}


bring_up_linux() {
    # Set bash internal field separator to comma
    TEMP_IFS=$IFS
    IFS=$(echo -en ",")

    if [ $IMM_DEVICE_COUNT -gt 0 ] 
    then
        find_interface $MAC_ADDR
        if [ $IS_XEN == 0 ]
        then
        #We attempt to find the bridged interface after the normal usb interface so that the functions following
        #this one can use the IS_XEN value that is set in find_bridged_connection to determine whether to use
        #$INTERFACE or $BR_INTERFACE when writing the file.
            find_bridged_connection $MAC_ADDR
        fi

	bring_up_cdc
    fi
    
    IFS=$TEMP_IFS
}


bring_up_vmware_3() {
    # Set bash internal field separator to comma
    TEMP_IFS=$IFS
    IFS=$(echo -en ",")

    #We keep the for loop here because vmware 3 uses dmesg to find the interface, and because of this
    #it is possible to get multiple instances of the same interface back.

    vmware_3_find_interface $MAC_ADDR

    if [ $IMM_ALREADY_UP == 1 ] #Can't talk to IMM
    then
        echo "Unable to communicate with the IMM, configuring interface..."

        echo "DHCP IP addresssing being set up..."
        check_for_previous_config_file
        write_config_file_dhcp
        set_driver_alias_for_network_restart
        bring_up_cdc

        if [ $? == 1 ]
        then
            echo "DHCP setup failed, setting up static address."
            #Set the netmask broader in case of autoconf
            CDC_ETHER_NETMASK=$CDC_ETHER_TEMP_NETMASK
            check_for_previous_config_file
            write_config_file_static
            set_driver_alias_for_network_restart
            disable_zeroconf
            delete_link_local_routes
            bring_up_cdc
            if [ $? == 1 ]
            then
                echo "Static setup failed."
                exit 4
            fi
        fi
    fi
    echo_interface_info

    #we can break here because only single node is supported
    #and setup worked or the connection was already enabled.
}

bring_up_vmware_4(){
    # No need for write_config_file on VMWare 4, esxcfg-vswitch and esxcfg-vswif write their own configs
    # No need for disable_zeroconf on VMWare 4, esxcfg-vswif takes care of the unneeded link-local routes

    #Look for previous installations of the devices that use the IBM_CDC_PG port group and delete them
    vmware4_delete_previous_interfaces
    echo "Pausing for 5 seconds to let system quiesce..."
    sleep 5


    echo "#IMM_DRIVER_STATUS=1" > $VMWARE_4_TEMP_FILE

    find_interface $MAC_ADDR

    echo "Bringing up interface via DHCP"
    vmware_4_bring_up_cdc DHCP
    if [ $? == 1 ]
    then
        CDC_ETHER_NETMASK=$CDC_ETHER_TEMP_NETMASK
        vmware_4_bring_up_cdc
    fi

    echo_interface_info
    vmware_4_imm_already_up_get_vswif_interface
}

bring_up_vmware() {
    # Set bash internal field separator to comma
    TEMP_IFS=$IFS
    IFS=$(echo -en ",")

    if [ $VER == "3" ]
    then
        bring_up_vmware_3
    elif [ $VER == 4 ]
    then
        bring_up_vmware_4
    fi
    IFS=$TEMP_IFS
}


bring_up_cdc_by_mac(){
    #If the interface can ping  IMM  --> Do Nothing
    #If the interface is up and more than 1 IMM shows --> re-write all config files for DHCP (Major OS != VMWARE --> if DHCP fails attempt to
    #   ping individual subnets and statically address.)
    #If the interface is down and only 1 IMM shows --> DHCP setup, unless Static setup flag specified. (Major OS != VMWARE --> If DHCP fails,
    #   attempt to ping individual subnets and statically address.)
    #If the interface is down and more than 1 IMM shows --> DHCP setup for all config files. (Major OS != VMWARE --> If DHCP fails attempt to
    #   ping individual subnets and statically address.)

    #check if MAC specifyed is one of our usb interface
    check_if_usbmac_exist $MAC_ADDR
    if [ $? != 0 ]
    then
        echo "cannot find mac as usb interface." >&1
    	exit 8
    fi	
    ######
    # If the driver does not exist, we need to install in hand
    ######
    if [ $IMM_DRIVER_STATUS == 0 ]
    then
        load_driver
    fi

    #delete_link_local_routes
    if [ $MAJOR_OS != VMWARE ]
    then
        bring_up_linux
    else
        bring_up_vmware
    fi

    # Drive some data over CDC interface.
    # (IMM hardware switches from "OS Booting" to "OS Booted" when traffic is seen on this interface in MCP.)
	
    # check usb network interface status
    # ping_imm_via_mac 	
    # if [ $? != 0 ]
    # then
    #      echo "Not able to successfully ping the IMM via MAC."
    #     exit 3
    #  fi

    #Ping reached IMM successfully, exit with Success
    echo "Exiting Successfully."
    exit 0
}

check_if_usbmac_exist()
{
	
    CUR_MAC=$1
    TEMP_IFS=$IFS
    IFS=$(echo -en ",")
    exist=1
	
    echo "begin to find if MAC " $CUR_MAC " exist..."
	
    for MAC in  $IMM_MACADDR_STRING
    do
        MAC=`echo $MAC | tr '[:lower:]' '[:upper:]'`
        if [ "$CUR_MAC" == "$MAC" ]
        then
            echo "success to find mac "$CUR_MAC
            exist=0
            break
        fi	
    done
	
    if [ "$exist" == "1" ]
    then
        echo "fail to find mac "$CUR_MAC
    fi	

    IFS=$TEMP_IFS
    return $exist
}

ip_command()
{
    if [ $# -eq 0 ]
    then
        if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
        then
            ip addr
        else
            ifconfig
	fi
    elif [ $# -eq 2 ]
    then
        if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
        then
    	    if [ "$2" == "up" ] || [ "$2" == "down" ]
	        then	
                ip link set $1 $2
	        else 
	            ip addr add $2/24 dev $1
	        fi
        else
            if [ "$2" == "up" ] || [ "$2" == "down" ]
            then	
                ifconfig $1 $2
            else 
                ifconfig $1 $2 netmask 255.255.255.0
            fi
	    fi
    fi
}

GET_IPADDR()
{
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
    then
        return `ip addr show $1 | awk 'NR==3{print $2}' | cut -c 6-14`
    else
        return `ifconfig $1 | awk 'NR==2{print $2}' | cut -c 6-20`
    fi 
}

GET_NETMASK()
{
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
    then
        "55.255.0"
    else
        return `ifconfig $1 | awk 'NR==2{print $4}' | cut -c 6-20`
    fi 
}

GET_MAC()
{
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
    then
        return `ip addr show $1 | grep ether | awk '{print $2}'`
    elif [ "$MAJOR_OS" == "RHEL" ] && [ $VER -eq 7 ]
    then 
        return `ifconfig $1 | grep ether | awk '{print $2}'`
    else
        return `ifconfig $1 | awk 'NR==1{print $5}'`
    fi 
}

route_command()
{
    if [ $IP_CMD_EXIST -eq 1 ] && [ $IP_CMD_CAN_BE_USED -eq 1  ]
    then
        ip route
    else
	route -n
    fi
}

################################ MAIN ######################################
    ARGS=$@
    ARGS_COUNT=$#

    # Start the log
    echo -en "\n\n"  `date` "\n" 
    echo cdc_interface VER 1.0.6

    if [ "$ARGS" == "-h" ] || [ "$ARGS" == "--help" ]
    then
        echo "--staticip  MAC    static bring up interface specified by MAC"
        echo "--restore          bring down all cdc interface"
        echo "--bringdown MAC    bring down interface specified by MAC"
        echo "--num FILE         get usb device count, and write MACs to FILE"
        echo "--bringup MAC  IP    bring up interface specified by MAC, give it a default IP"
        echo "--nodes nodes      check nodes number"
        echo "--addip INTERFACE IP [MASK]   add ip address for the interface"
        echo "--delip INTERFACE IP   delete ip address for the interface"
        exit 0
    fi		

    detect_os_variant

    #Check if driver is currently loaded before doing anything else
    get_driver_status

    #Unload driver, then reload it to make sure it loads properly for VMWare, then get MAC addresses
    get_mac_addr

    #Parse the command line and execute as instructed
    parse_command_line_args $ARGS

    #bring_up_all_cdc

    exit 0

