#==========================================================================
#
# Install a Microsoft Language Pack
#
# AUTHOR: Dennis Span
# DATE  : 27.08.2018
#
# COMMENT: This script installs a Microsoft language pack (MUI) on the local machine
#
# Note: see the article 'https://dennisspan.com/powershell-scripting-template-for-sccm-packages/' for a detailed description how to use this template
#
# Note: for an overview of all functions in the PowerShell function library 'DS_PowerShell_Function_Library.psm1' see:
#       -Windows functions: https://dennisspan.com/powershell-function-library/powershell-functions-for-windows/
#       -Citrix functions: https://dennisspan.com/powershell-function-library/powershell-functions-for-citrix/
#          
# Change log:
# -----------
# <DATE> <NAME>: <CHANGE DESCRIPTION>
#==========================================================================

# Get the script parameters if there are any
param
(
    # The only parameter which is really required is 'Uninstall'
    # If no parameters are present or if the parameter is not
    # 'uninstall', an installation process is triggered
    [string]$Installationtype
)

# define Error handling
# note: do not change these values
$global:ErrorActionPreference = "Stop"
if($verbose){ $global:VerbosePreference = "Continue" }

############################
# Preparation              #
############################

# Disable File Security
$env:SEE_MASK_NOZONECHECKS = 1

# Custom variables [edit]
$BaseLogDir = "C:\Logs"                                        # [edit] add the location of your log directory here
$PackageName = "Microsoft Language Pack Spanish"               # [edit] enter the display name of the software (e.g. 'Microsoft Language Pack Spanish' or 'Microsoft MUI German')

# Global variables
$StartDir = $PSScriptRoot # the directory path of the script currently being executed
if (!($Installationtype -eq "Uninstall")) { $Installationtype = "Install" }
$LogDir = (Join-Path $BaseLogDir $PackageName).Replace(" ","_")
$LogFileName = "$($Installationtype)_$($PackageName).log"
$LogFile = Join-path $LogDir $LogFileName

# Create the log directory if it does not exist
if (!(Test-Path $LogDir)) { New-Item -Path $LogDir -ItemType directory | Out-Null }

# Create new log file (overwrite existing one)
New-Item $LogFile -ItemType "file" -force | Out-Null

# Import the Dennis Span PowerShell Function Library
Import-Module "C:\Scripts\DS_PowerShell_Function_Library.psm1"

DS_WriteLog "I" "START SCRIPT - $Installationtype $PackageName" $LogFile
DS_WriteLog "-" "" $LogFile

############################
# Installation             #
############################

# Set variables
$FileName = "x64fre_Server_es-es_lp.cab"                                # [edit] enter the name of the installation file (e.g. 'x64fre_Server_de-de_lp.cab')
$FileSubfolder = "Files"                                                # [edit] enter the name of the subfolder which contains the installation file

# Install the language pack (using the log file defined in the variable $LogFile)
$Language = $FileName.Substring(14,5)                                   # Retrieve the language code from the Windows Language Pack file name (e.g. 'de-de' for Germany/German)
$FileSubfolder = "Files\$($Language)"                                   # Add the name of the language folder (e.g. 'es-es' or 'de-de') to the file subfolder path (the result will be something like 'Files\es-es' or 'Files\de-de')
$FullPath = Join-Path $StartDir $FileSubfolder                          # Concatenate the two directories $StartDir and $InstallFileFolder
$FileFullPath = Join-Path $FullPath $FileName                           # Concatenate the two directories $StartDir and $InstallFileFolder
$LogFileMUI = Join-Path $LogDir "DISM_Log_LanguagePack_$($Language.ToUpper()).log"
DS_WriteLog "I" "Install the Windows Language Pack $($Language.ToUpper())" $LogFile
DS_WriteLog "I" "-File full path: $($FileFullPath)" $LogFile
DS_WriteLog "I" "-DISM log file: $($LogFileMUI)" $LogFile
Try {
    DS_WriteLog "I" "-Command: Add-WindowsPackage -Online -PackagePath ""$FileFullPath"" -LogPath ""$LogFileMUI"" -NoRestart" $LogFile
    Add-WindowsPackage -Online -PackagePath $FileFullPath -LogPath $LogFileMUI -NoRestart
    DS_WriteLog "S" "The language pack $($Language.ToUpper()) was installed successfully" $LogFile
} Catch {
    DS_WriteLog "E" "Failed to install the language pack $($Language.ToUpper())" $LogFile
    Exit 1
}

############################
# Finalize                 #
############################

# Enable File Security  
Remove-Item env:\SEE_MASK_NOZONECHECKS

DS_WriteLog "-" "" $LogFile
DS_WriteLog "I" "End of script" $LogFile