// Copyright © 2023 - present Lenovo.  All rights reserved.
// Confidential and Proprietary.


Server_Pool = function () {

    let log = console.log;

    let vue_app = undefined;

    let crypt_temp_file = top.pathJoin(top.basePath(), "multi_server.json");

    let PASSWORD_NEED_RESET = 54;

    let filter_file = "multi_server_management_support_list.json";

    let cfc_support_list = {
        data: [],
        includes: function (mt) {
            for (let item of this.data) {
                if (item.mt === mt) {
                    return true;
                }
            }
            return false;
        }
    }

    function init() {
        log("init server pool");
        cfc_support_list.data = top.loadSetting(filter_file);
        checkNextEnable();
        init_vue();
    }

    function next() {
        if (vue_object.selectedCount > 10) {
            dialog.warning.showDialog();
            return false;
        }
        if (!vue_object.isAllReady) {
            checkAllState();
            return false;
        }
        Server_Pool.selectedServerList = [];
        let mt = new Set();
        let ipset = {};
        for (let item of vue_object.serverData) {
            if (item.selected) {
                Server_Pool.selectedServerList.push(
                    {
                        ip: item.ip,
                        port: item.port,
                        userName: item.userName,
                        mt: item.mt,
                        sn: item.sn
                    }
                );
                if (item.internalIP in ipset) {
                    vue_object.errorInfo = tr("Internal IP conflict between {0} with {1}, possibly the same machine").format(item.ip, ipset[item.ip]);
                    ElementPlus.ElMessageBox.alert(vue_object.errorInfo);
                    return false;
                }
                ipset[item.internalIP] = item.ip;
                mt.add(item.mt);
            }
        }
        top.gl_multi_server_management = Server_Pool.selectedServerList.length > 1;
        if (mt.size == 1) {
            Banner.set_active_status(Array.from(mt)[0], "");
        } else {
            Banner.set_active_status(tr("Multi"), "");
        }
        return true;
    }

    function previous() {
        top.toggleNext(true);
        return true;
    }

    function validateIpAddr(rule, value, callback) {
        if (top.ipValidate(bmcInfo.ipFrom)) {
            callback();
        } else {
            callback("IP Address format error!")
        }
    }

    const rules = Vue.reactive({
        ipAddr: [{ validator: validateIpAddr, trigger: 'blur' }],
        userName: [{
            validator: function (rule, value, callback) {
                if (bmcInfo.userName) {
                    callback();
                } else {
                    callback("User name validation error.");
                }
            },
            trigger: 'blur',
        }],
        password: [{
            validator: function (rule, value, callback) {
                if (bmcInfo.password) {
                    callback();
                } else {
                    callback("Password validation error.");
                }
            },
            trigger: 'blur',
        }]
    });

    exportData = Vue.reactive({
        fileName: top.getSupportDir() + "server_list.csv",
        loading: false,
        data: {}
    });

    async function runScan(ip, port, userName) {
        deleteFile(top.getSupportFile("commonResult", null, ip));
        let args = [
            top.getSystemFile("oneCli"),
            "update",
            "scan",
            "--bmc",
            getConnectionInfo(ip, port, userName),
            "--never-check-trust",
            "--quiet",
            ...top.add_output_log5(ip)
        ];
        top.printLog("INFO", "run command:" + getRunProgramCommend(args));
        let result = await top.runProgram(args);
        showOneCliErrorInfo(result, ip);
        fetchDataFromScan(result, ip);
        return result;
    }

    function fetchDataFromScan(result, ip) {
        try {
            if (!result.success) {
                return;
            }
            let doc = top.readTextFile(top.getSupportFile("scanResult", "", ip));
            let dom = ezJsLib.XmlDom.parse(doc);
            let packageList = dom.documentElement.getElementsByTagName("PACKAGE");
            for (let j = 0; j < packageList.length; ++j) {
                let pkg = packageList[j];
                let name = ezJsLib.XmlDom.getValueByTagFromNode("NAME", pkg);
                let version = ezJsLib.XmlDom.getValueByTagFromNode("VERSION", pkg);
                exportData.data[ip][name] = version;
            }
        } catch (exp) {
            printLog("Exception: ", exp);
            log("Exception: ", exp);
        }
    }

    async function export_config_batch_file() {
        var oneCliList = [];
        exportData.data = {};
        for (var i = 0; i < vue_object.serverData.length; i++) {
            var item = vue_object.serverData[i];
            if (!item.selected) {
                continue;
            }
            exportData.data[item.ip] = {
                "ip": item.ip,
                "userName": item.userName,
                "mt": item.mt,
                "sn": item.sn,
                "BMC": "",
                "UEFI": "",
                "FPGA": ""
            };
            if (item.value) {
                var asyncResult = runScan(item.ip, item.port, item.userName);
            }
            oneCliList.push(asyncResult);
            if (oneCliList.length >= 10) {
                await oneCliList[0];
                oneCliList.splice(0, 1);
            }
        }
        for (let item of oneCliList) {
            await item;
        }
    }

    async function onExportToExcel() {
        exportData.loading = true;
        await export_config_batch_file();

        if (exportData.fileName.endsWith(".xls")) {
            var fileContent = "IP\tUser Name\tMT\tSN\tBMC Version\tUEFI Version\tFPGA Version\n";
            for (let key in exportData.data) {
                var item = exportData.data[key];

                fileContent += "{0}\t{1}\t{2}\t{3}\t{4}\t{5}\t{6}\n".format(item.ip, item.userName, item.mt, item.sn, item.BMC, item.UEFI, item.FPGA);
            }
        } else {
            var fileContent = "IP,User Name,MT,SN,BMC Version,UEFI Version,FPGA Version\n";
            for (let key in exportData.data) {
                var item = exportData.data[key];
                fileContent += "{0},{1},{2},{3},{4},{5},{6}\n".format(item.ip, item.userName, item.mt, item.sn
                    , item.BMC
                    , item.UEFI
                    , item.FPGA);
            }
        }
        
        var result = top.writeTextFile(exportData.fileName, fileContent, false);
        top.printLog("Write data to file ", result == true ? "Success" : "Failure");
        exportData.loading = false;
        return result;
    }

    async function init_vue() {
        await refresh()
        if (vue_app != undefined) {
            return;
        }
        vue_app = Vue.createApp(vue_class);
        vue_app.testValue = 9;
        for ([iconName, comp] of Object.entries(ElementPlusIconsVue)) {
            vue_app.component(iconName, comp);
            log(iconName, comp);
        }
        vue_app.use(ElementPlus);
        vue_app.mount('#server_pool_vue');
    }

    function uninit() {
        if (vue_app != undefined) {
            vue_app.unmount();
            vue_app = undefined;
        }
    }

    let multiRunning = Vue.reactive({
        runningCount: 0,
        delayCount: 0,
        finished: 0,
        passed: 0,
        failed: 0,
        scanningCount: 0,
        total: 0,
        scanningTotal: 0,
        idList: new Set(),
        reset: function () {
            this.runningCount = 0;
            this.delayCount = 0;
            this.finished = 0;
            this.passed = 0;
            this.failed = 0;
            this.scanningCount = 0;
            this.scanningTotal = 0;
            this.total = 0;
            this.idList = new Set();
        }
    });

    let ui = Vue.reactive({
        credential_show: false
    })

    async function ChangePassword(ip, port, userName, newPassword) {
        resetAllErrorInfo();
        deleteFile(top.getSupportFile("commonResult", null, ip));
        let args = [
            top.getSystemFile("oneCli"),
            "misc",
            "bmcpassword",
            "--bmc",
            getConnectionInfo(ip, port, userName),
            "--never-check-trust",
            "--newpwd",
            newPassword,
            "--quiet",
            ...add_output_log5(ip)
        ];
        printLog("INFO", "run command:" + getRunProgramCommend(args));
        let result = await runProgram(args);
        let key = userName + "@" + ip + (port ? ":" + port : "");
        let index = indexOfServer(key);
        showOneCliErrorInfo(result, ip, index);
        if (!result.success && result.errorCode == 1) {
            vue_object.serverData[index].scanResult = tr("Fail");
            vue_object.serverData[index].state = tr("Error");
            vue_object.serverData[index].errorInfo = tr("Change password failed.");
            vue_object.serverData[index].needChangePassword = false;
            onFilterStringChanged(vue_object.filterString);
        } else {
            multi_config.AppendOrUpdate(ip, null, null, null, null, null, null, false);
        }
        return result.success;
    }

    async function GetCert(ip, port, index) {
        resetAllErrorInfo();
        deleteFile(top.getSupportFile("commonResult", null, ip));
        let certIMMSaveFile = "IMM_" + ip + ".txt";
        certIMMSaveFile = top.pathJoin(top.getTempPath4ThisTask(), ip, certIMMSaveFile);
        let certIMMTempFile = top.pathJoin(top.getTempPath4ThisTask(), ip, "IMM_Cert.txt");
        top.deleteFile(certIMMTempFile);
        let args = [
            top.getSystemFile("oneCli"),
            "misc",
            "gettrust",
            "--host",
            ip,
            "--type",
            "https",
            "--result",
            certIMMTempFile,
            ...add_output_log5(ip),
            ...port ? ['--port', port] : []
        ];
        printLog("INFO", "run command:" + getRunProgramCommend(args));
        multiRunning.scanningCount++;
        let result = await runProgram(args);
        multiRunning.scanningCount--;
        if (!result.success) {
            showOneCliErrorInfo(result, ip, index, tr("Get cert failed"));
        }
        if (!top.fileExists(certIMMTempFile)) {
            showOneCliErrorInfo(result, ip, index, tr("Get cert failed"));
            result.success = false;
        }
        if (!result.success) {
            vue_object.errorInfo = vue_object.errorInfo;
            vue_object.errorInfo = "";
        }
        let savedCert = top.readTextFile(certIMMSaveFile);
        let currentCert = top.readTextFile(certIMMTempFile);
        if (savedCert == null) {
            top.writeTextFile(certIMMSaveFile, currentCert);
            savedCert = currentCert;
        }
        if (currentCert != null) {
            result.success = currentCert == savedCert;
            if (!result.success) {
                vue_object.errorInfo = tr("The certificate is different from the one previously saved");
            }
        } else {
            result.success = false;
        }
        return {
            ok: result.success,
            pem: currentCert
        }
    }

    let dialog = Vue.reactive({
        bmcCertificate: {
            show: false,
            certText: "",
            ip: "",
            port: null,
            userName: "",
            index: -1,
            ok: async function () {
                this.show = false;
                let certIMMSaveFile = "IMM_" + this.ip + ".txt";
                certIMMSaveFile = top.pathJoin(top.getTempPath4ThisTask(), this.ip, certIMMSaveFile);
                if (this.certText) {
                    top.writeTextFile(certIMMSaveFile, this.certText);
                }
                vue_object.serverData[this.index].cert.acceptedByUser = true;
                resetAllErrorInfo();
                vue_object.serverData[this.index].state = "";
                vue_object.serverData[this.index].scanResult = "";
                vue_object.onecliIsRunning = true;
                let result = await doScan(this.ip, this.port, this.userName, this.index);
                vue_object.onecliIsRunning = false;
                if (result.success) {
                    vue_object.serverData[this.index].state = tr("Ready");
                    vue_object.serverData[this.index].scanResult = tr("Pass");
                    vue_object.serverData[this.index].errorInfo = "";
                    if (result.errorCode == PASSWORD_NEED_RESET) {
                        vue_object.serverData[this.index].errorInfo = tr("The password needs to be changed");
                        vue_object.serverData[this.index].state = tr("Warning");
                        vue_object.serverData[this.index].needChangePassword = true;
                        vue_object.serverData[this.index].scanResult = tr("Fail");
                    }
                } else {
                    showOneCliErrorInfo(result, this.ip, this.index);
                    vue_object.serverData[this.index].state = tr("Error");
                    vue_object.serverData[this.index].scanResult = "Fail";
                }
                onFilterStringChanged();
                this.resolve("Ok");
            },
            cancel: function () {
                this.show = false;
                this.resolve("Cancel");
            },
            showDialog: async function () {
                this.show = true;
                let promise = new Promise((resolve) => {
                    this.resolve = resolve;
                });
                return promise;
            }
        },
        batchCertificate: {
            show: false,
            "text": tr("The current server certificate is different from the previously saved certificate. Accept the new certificate?"),
            ok: async function () {
                this.show = false;
                for (let item of vue_object.serverData) {
                    if (!item.cert.ok && !item.cert.acceptedByUser) {
                        let certIMMSaveFile = "IMM_" + item.ip + ".txt";
                        certIMMSaveFile = top.pathJoin(top.getTempPath4ThisTask(), item.ip, certIMMSaveFile);
                        if (item.cert.pem) {
                            top.writeTextFile(certIMMSaveFile, item.cert.pem);
                        }
                        item.cert.acceptedByUser = true;
                        item.state = "";
                        item.scanResult = "";
                    }
                }
                onFilterStringChanged();
                this.resolve(true);
            },
            cancel: function () {
                this.show = false;
                this.resolve(false);
            },
            showDialog: async function () {
                let promise = new Promise((resolve) => {
                    this.resolve = resolve;
                });
                this.show = true;
                return promise;
            }
        },
        changePassword: {
            show: false,
            text: "",
            password: "",
            confirmPassword: "",
            ip: "",
            userName: "",
            showDialog: async function () {
                this.show = true;
                this.showPassword = false;
                this.showConfirmPassword = false;
                let promise = new Promise((resolve) => {
                    this.resolve = resolve;
                });
                return promise;
            },
            onConfirm: async function () {
                resetAllErrorInfo();
                vue_object.onecliIsRunning = true;
                vue_object.scanMessage = "Change password...";
                let result = await ChangePassword(this.ip
                    , this.port
                    , this.userName
                    , this.password
                );
                if (result) {
                    result = await encrypt(this.ip, this.port, this.userName, this.password);
                    if (!result) {
                        if (this.resolve) {
                            this.resolve(false);
                        }
                        return;
                    }
                    result = await doScan(this.ip,
                        this.port,
                        this.userName);
                    if (result.success) {
                        this.show = false;
                        vue_object.serverData[this.index].needChangePassword = false;
                        vue_object.serverData[this.index].password = this.password;
                        vue_object.serverData[this.index].scanResult = tr("Pass");
                        vue_object.serverData[this.index].state = tr("Ready");
                        vue_object.serverData[this.index].errorInfo = "";
                    } else {
                        vue_object.serverData[this.index].needChangePassword = true;
                    }
                    onFilterStringChanged(vue_object.filterString);
                }
                vue_object.onecliIsRunning = false;
                if (this.resolve) {
                    this.resolve(false);
                }
            },
            onCancel: function () {
                if (this.resolve) {
                    this.resolve(false);
                }
                this.show = false;
            }
        },
        credential: {
            show: false,
            text: tr("Input the common BMC credentials"),
            userName: "",
            password: "",
            port: "",
            ip: "",
            onOk: function () {
                this.show = false;
                this.resolve(true)
            },
            onCancel: function () {
                this.show = false;
                this.resolve(false)
            },
            showDialog: async function () {
                this.show = true;
                let result = new Promise((resolve) => {
                    this.resolve = resolve;
                });
                return result;
            }
        },
        warning: {
            show: false,
            text: "",
            showDialog: async function (msg = tr("Batch management is limited to 10 devices. Adjust your selection accordingly.")) {
                this.text = msg;
                let promise = new Promise((resolve) => {
                    this.resolve = resolve;
                });
                this.show = true;
                return await promise;
            },
            onOk: function () {
                this.show = false;
                this.resolve(true);
            }
        },
        changePasswordBatch: {
            show: false,
            password: "",
            confirmPassword: "",
            showDialog: async function () {
                this.show = true;
                this.password = "";
                this.confirmPassword = "";
                this.showPassword = false;
                this.showConfirmPassword = false;
                let promise = new Promise((resolve) => {
                    this.resolve = resolve; 
                });
                return promise;
            },
            check: async function () {
                let tempNeedChangePassword = false;
                for (let i = 0; i < vue_object.serverData.length; i++) {
                    let item = vue_object.serverData[i];
                    if (item.selected && item.needChangePassword) {
                        tempNeedChangePassword = true;
                        break;
                    }
                }
                if (!tempNeedChangePassword) {
                    return false;
                }
                return this.showDialog();
            },
            onConfirm: async function () {
                let procList = [];
                vue_object.onecliIsRunning = true;
                resetAllErrorInfo();
                vue_object.scanMessage = tr("Changing password ...");
                let testConnect = async function (ip, port, userName, password, index) {
                    result = await encrypt(ip, port, userName, password);
                    if (!result) {
                        return false;
                    }
                    result = await doScan(ip, port, userName, index);
                    if (result.success) {
                        vue_object.serverData[index].needChangePassword = false;
                        vue_object.serverData[index].password = password;
                        vue_object.serverData[index].scanResult = tr("Pass");
                        vue_object.serverData[index].state = tr("Ready");
                        vue_object.serverData[index].errorInfo = "";
                        multi_config.AppendOrUpdate(ip, null, null, null, null, null, null, false);
                    } else {
                        vue_object.serverData[index].needChangePassword = true;
                    }
                    onFilterStringChanged(vue_object.filterString);
                    return result.success;
                }
                let allSuccess = true;
                for (let index = 0; index < vue_object.serverData.length; index++){
                    let item = vue_object.serverData[index];
                    if (!item.selected) {
                        continue;
                    }
                    if (!item.needChangePassword) {
                        continue;
                    }
                    let proc = ChangePassword(item.ip, item.port, item.userName, this.password);
                    let newItem = clone(item);
                    newItem.password = this.password;
                    newItem.index = index;
                    procList.push({
                        item: newItem,
                        proc: proc
                    });
                    if (procList.length > 10) {
                        let result = await procList[0].proc;
                        procList.splice(0, 1);
                        if (result) {
                            let val = await testConnect(procList[0].item.ip, procList[0].item.port
                                , procList[0].item.userName, procList[0].item.password
                                , procList[0].item.index);
                            allSuccess &= val;
                        } else {
                            allSuccess = false;
                        }
                    }
                }
                for (let item of procList) {
                    let result = await item.proc;
                    if (result) {
                        let val = await testConnect(item.item.ip, item.item.port
                            , item.item.userName, item.item.password
                            , item.item.index);
                        allSuccess &= val;
                    } else {
                        allSuccess = false;
                    }
                }
                vue_object.onecliIsRunning = false;
                this.show = !allSuccess;
                if (this.resolve) {
                    this.resolve(allSuccess);
                }
                return allSuccess;
            },
            onCancel: function () {
                if (this.resolve) {
                    this.resolve(false);
                }
                this.show = false;
            }
        }
    });

    let bmcInfo = Vue.reactive({
        ipFrom: "",
        ipFromFormatError: true,
        userName: "",
        password: "",
        port: "",
        portFormatError: false,
        ipPrefix: "",
        ipSuffix: "",
        scanResult: "",
        scanResultDetail: "",
        ipTo: function () { return this.ipPrefix + this.ipSuffix },
        verifyIP: function () {
            return top.ipValidate(this.ipFrom);
        }
    });

    let findInfo = Vue.reactive({
        total: 0,
        passCount: 0,
        failCount: 0,
        scannedCount: 0,
        discoverCount: 0,
        useDiscover: false,
        reset: function () {
            this.total = 0;
            this.passCount = 0;
            this.failCount = 0;
            this.scannedCount = 0;
            this.discoverCount = 0;
            this.useDiscover = false;
        }
    });

    let vue_object = Vue.reactive({
        serverData: [],
        serverDataDisplay: [],
        serverDataFromFile: [],
        filterString: "",
        checkAllIndeterminate: false,
        allChecked: false,
        onecliIsRunning: false,
        errorInfo: "",
        errorInfoDetail: "",
        statusInfo: "",
        showErrorInfoDetail: false,
        warningInfo: "",
        warningInfoDetail: "",
        showWarningInfoDetail: false,
        errorCode: 0,
        selectedCount: 0,
        passwordIsVisible: false,
        headerVisible: false,
        confirmDialogVisible: false,
        confirmDialogMessage: tr("Are you sure to delete the selected server?"),
        multi_server_config_file: top.pathJoin(top.basePath(), "multi_server_config_file.json"),
        tipsInfo: "",
        scanResultDialog: false,
        scanResultInfo: "",
        currentIndex: undefined,
        isAllReady: false,

        "getKey": function (item) {
            return [
                item.userName,
                "@",
                item.ip,
                ...[item.port ? `:${item.port}` : '']
            ].join("");
        },

        filtered: function (id) {
            if (id < 0 || id > vue_object.serverData.length) {
                return true;
            }
            if (vue_object.filterString === "") return false;
            let value = vue_object.serverData[id];
            let needFilter = false;
            needFilter |= value.mt.toLowerCase().includes(vue_object.filterString.toLowerCase());
            needFilter |= value.ip.toLowerCase().includes(vue_object.filterString.toLowerCase());
            needFilter |= value.sn.toLowerCase().includes(vue_object.filterString.toLowerCase());
            needFilter |= value.state.toLowerCase().includes(vue_object.filterString.toLowerCase());
            return !needFilter;
        },
        testClick: function (scopt) {
            console.log(scopt);
        },
        filterHandler: function (value, row, column) {
            return filtered(row);
        }
    });

    let refresh = async function () {
        for (let item of multi_config.Data().ENCRYPTED_INFO) {
            if (!cfc_support_list.includes(item.mt)) {
                continue;
            }
            await appendServer(item.ip, item.internalIP, item.port
                , item.userName, ""
                , item.mt, item.sn
                , multi_config.GetKey(item)
                , item.value, false, { ok: true, pem: "" }, item.needChangePassword
            );
        }
    }

    let checkNextEnable = function () {
        let allUnSeled = true;
        for (var i = 0; i < vue_object.serverData.length; i++) {
            let value = vue_object.serverData[i];
            if (value.selected) {
                allUnSeled = false;
                break;
            }
        }
        top.toggleNext(!allUnSeled);
    }

    let appendServer = async function (ip, internalIP, port, userName, passwd, mt, sn, key, value, selected, cert = { ok: true, pem: "" }, needChangePassword = false) {
        let isExists = false;
        multi_config.AppendOrUpdate(ip, internalIP, port, userName, value, mt, sn);
        for (let j = 0; j < vue_object.serverData.length; j++) {
            if (vue_object.serverData[j].ip === ip) {
                isExists = true;
                if (userName && value && userName.length > 0 && value.length > 0) {
                    vue_object.serverData[j].userName = userName;
                    vue_object.serverData[j].value = value;
                }
                break;
            }
        }
        if (isExists) {
            return;
        }
        vue_object.serverData.push({
            "ip": ip,
            "internalIP": internalIP,
            "port": port,
            "ipDisplay": ip,
            "mt": mt,
            "mtDisplay": mt,
            "sn": sn,
            "snDisplay": sn,
            "value": value,
            "selected": !!selected,
            "mac": "",
            "userName": userName,
            "password": passwd,
            "edit": false,
            "scanResult": selected ? tr("Pass") : "",
            cert: cert,
            needChangePassword: needChangePassword,
            errorInfo: "",
            errorInfoDetail: "",
            state: (cert.ok && !needChangePassword) ? "" : tr("Warning"),
            stateDisplay: (cert.ok && !needChangePassword) ? "" : tr("Warning")
        });
        if (userName != '' && passwd != '') {
            await encrypt(ip, port, userName, passwd);
        }
        onFilterStringChanged(vue_object.filterString);
        onCheckItemChange(vue_object.serverData.length - 1);
    }

    let resetAllErrorInfo = function () {
        vue_object.errorInfo = "";
        vue_object.errorInfoDetail = "";
        vue_object.scanResultInfo = "";
        vue_object.statusInfo = "";
        findInfo.total = 0;
        findInfo.passCount = 0;
        findInfo.failCount = 0;
    }

    async function encrypt(ip, port, userName, password, index) {
        if (!userName || !password || !ip) {
            return false;
        }
        let multiData = {
            connection_credentials: {
                "default_alias": userName + ":" + password
            },
            target_systems: [
                ip + (port ? ":" + port : "")
            ]
        }
        try {
            top.writeTextFile(crypt_temp_file, JSON.stringify(multiData, null, 2));
            let writeEncryptFile = function () {
                let cryptObj = {
                    MODE: vue_object.MODE,
                    RAND_NUM: vue_object.RAND_NUM,
                    ENCRYPTED_INFO: []
                }
                if (cryptObj.MODE && cryptObj.RAND_NUM && cryptObj.MODE.length > 0 && cryptObj.RAND_NUM.length > 0) {
                    top.writeTextFile(top.getSystemFile("crypt"), JSON.stringify(cryptObj, null, 2));
                }
            }
            if (!top.fileExists(top.getSystemFile("crypt"))) {
                if (vue_object.RAND_NUM != undefined && vue_object.MODE != undefined) {
                    writeEncryptFile();
                }
            }

            let args = [];
            args.push(top.getSystemFile("oneCli"));
            args.push("encrypt");
            args.push("--configfile");
            args.push(crypt_temp_file);
            args.push("--unattended");
            args.push("-q");
            args.push(...add_output_log5(ip));
            top.store_onecli_command(args);
            printLog("INFO", "run command:" + getRunProgramCommend(args));
            top.deleteFile(top.getSupportFile("commonResult", null, ip));
            result = await runProgram(args, null);
            if (!result.success) {//if file format error occurred, try to restore it.
                top.deleteFile(top.getSupportFile("commonResult", null, ip));
                writeEncryptFile();
                result = await runProgram(args, null);
                if (!result.success) {
                    top.deleteFile(top.getSystemFile("crypt"));
                    result = await runProgram(args, null);
                }
            }
            log(result);
            if (!result.success) {
                resetAllErrorInfo();
                showOneCliErrorInfo(result, ip, index, tr("OneCli execute encrypt fail."));
            }
            top.deleteFile(crypt_temp_file);
            cryptObj = JSON.parse(top.readTextFile(top.getSystemFile("crypt")));
            vue_object.MODE = cryptObj.MODE;
            vue_object.RAND_NUM = cryptObj.RAND_NUM;
            multi_config.UpdateAttend(vue_object.MODE, vue_object.RAND_NUM);
            let key = userName + "@" + ip + (port ? ":" + port : "");
            let value = "";
            for (let i = 0; i < cryptObj.ENCRYPTED_INFO.length; i++) {
                let item = cryptObj.ENCRYPTED_INFO[i];
                if (key === item.key) {
                    value = item.value;
                    break;
                }
            }
            if (value === "") {
                vue_object.errorInfo = tr("Find encrypted password failed.");
                return;
            }
            if (index && index < vue_object.serverData.length) {
                vue_object.serverData[index].value = value;
            } else {
                for (let i = 0; i < vue_object.serverData.length; i++) {
                    let item = vue_object.serverData[i];
                    if (ip === item.ip) {
                        item.value = value;
                        break;
                    }
                }
            }
            if (result.success) {
                log("update encrypt data for ", ip);
                multi_config.AppendOrUpdate(ip, null, port, userName, value);
            }
            return result.success;
        } catch (e) {
            log(e);
        }
        return false;
    }

    async function onAddByOnecli(form) {
        let ip = bmcInfo.ipFrom;
        let ipEnd = bmcInfo.ipTo();
        if (!ipEnd) {
            ipEnd = ip;
        }
        if (ip == ipEnd) {
            addSingleIp = true;
        }
        resetAllErrorInfo();
        findInfo.reset();
        log(ip, ipEnd);
        let number = 1;
        if (!top.ipValidate(ip)) {
            vue_object.errorInfo = tr("Error: IP address format error, please check it.");
            $("#server-pool-input-ipaddr-from").focus();
            return;
        }
        function ipToInt(ipAddr) {
            let itemList1 = ipAddr.split(".");
            let val = 0;
            for (let item of itemList1) {
                val += val * 255 + parseInt(item);
            }
            return val;
        }
        number = ipToInt(ipEnd) - ipToInt(ip) + 1;
        if (ui.credential_show) {
            if (bmcInfo.userName === "") {
                vue_object.errorInfo = tr("Common.RquiredField");
                $("#server-pool-input-userName").focus();
                return;
            }
            if (bmcInfo.password === "") {
                vue_object.errorInfo = tr("Common.RquiredField");
                $("#server-pool-input-password").focus();
                return;
            }
        }

        let discoverFileName = top.getSupportFile("discoverResult");
        async function scanFunc() {
            deleteFile(top.getSupportFile("commonResult"));
            top.deleteFile(discoverFileName);
            let args = [];
            args.push(top.getSystemFile("oneCli"));
            args.push("discover");
            if (ip != "") {
                args.push("--ip");
                args.push(ip);
                args.push("--number");
                args.push(number);
                args.push("--nossdp");
                args.push("--noslpdiscover");
                args.push("--max-thread-number");
                args.push(1000);
                args.push("--filter-file");
                args.push(getConfigFileFullPath(filter_file));
            }
            // args.push("--nogp");
            args.push(...add_output_log5());
            top.store_onecli_command(args);
            printLog("INFO", "run command:" + getRunProgramCommend(args));
            result = await runProgram(args);
            return result.success;
        }
        async function readResult(isFinished) {
            if (!top.fileExists(discoverFileName)) {
                if (vue_object.onecliIsRunning) {
                    setTimeout(readResult, 1000);
                }
                return;
            }
            try {
                let rootObj = JSON.parse(top.readTextFile(discoverFileName));
                findInfo.total = rootObj.Total;
                findInfo.passCount = rootObj.Find;
                findInfo.failCount = rootObj.Skip;
                findInfo.discoverCount = rootObj.Discover;
                if (findInfo.useDiscover) {
                    vue_object.scanResultInfo = "Discovered: " + findInfo.discoverCount;
                } else {
                    vue_object.scanResultInfo = tr("Total scanned:") + " " + findInfo.total
                        + ", <span style='color:darkgreen'>"
                        + findInfo.passCount + " " + tr("matched") + "</span>"
                        + ", <span style='color:chocolate'>" + findInfo.failCount + " " + tr("skipped") + "</span>";
                }
                if (isFinished) {
                    for (let item of rootObj.List) {
                        if (!cfc_support_list.includes(item.MT)) {
                            continue;
                        }
                        if (!item.Internal_IP) {
                            item.Internal_IP = item.IP;
                        }
                        if (ui.credential_show) {
                            await appendServer(item.IP, item.Internal_IP
                                , "", bmcInfo.userName, bmcInfo.password, item.MT
                                , item.SN);
                        } else {
                            await appendServer(item.IP, item.Internal_IP
                                , "", "", "", item.MT
                                , item.SN);
                        }
                    }
                }
            } catch (e) {
                printLog(e);
            }
            if (vue_object.onecliIsRunning) {
                setTimeout(readResult, 1000);
            }
        }
        setTimeout(readResult, 1000);
        vue_object.onecliIsRunning = true;
        await scanFunc();
        vue_object.onecliIsRunning = false;
        readResult(true);
    }

    let indexOfServer = function (key) {
        let index = -1;
        for (let i = 0; i < vue_object.serverData.length; i++) {
            if (vue_object.getKey(vue_object.serverData[i]) === key) {
                index = i;
                break;
            }
        }
        return index;
    }

    function showOneCliErrorInfo(result, ip, index = -1, errorMessage = tr("Execute onecli command fail.")) {
        if (index != -1) {
            vue_object.serverData[index].errorInfo = "";
            vue_object.serverData[index].errorInfoDetail = "";
        }
        if (result.success) {
            return;
        }
        let errorInfoUpdated = false;
        if (!vue_object.errorInfo) {
            vue_object.errorInfo = errorMessage;
            errorInfoUpdated = true;
        }
        let commonResult = top.getSupportFile("commonResult", null, ip);
        let xmlContent = top.readTextFile(commonResult);
        if (!xmlContent) {
            if (errorInfoUpdated) {
                vue_object.errorInfoDetail = result.stderr;
            }
            if (index != -1) {
                vue_object.serverData[index].errorInfo = errorMessage;
                vue_object.serverData[index].errorInfoDetail = vue_object.errorInfoDetail;
            }
            return;
        }
        let comm_dom = ezJsLib.XmlDom.parse(xmlContent);
        if ("Yes" == ezJsLib.XmlDom.getValueByTagFromNode("ERROR", comm_dom.documentElement)) {
            if (errorInfoUpdated) {
                vue_object.errorInfoDetail = ezJsLib.XmlDom.getValueByTagFromNode("MESSAGE", comm_dom.documentElement);
            }
            vue_object.errorCode = ezJsLib.XmlDom.getValueByTagFromNode("ERRORCODE", comm_dom.documentElement);
            if (index != -1) {
                vue_object.serverData[index].errorInfo = errorMessage;
                vue_object.serverData[index].errorInfoDetail = vue_object.errorInfoDetail;
            }
        }
    }

    let doScan = async function (ip, port, userName, index) {
        resetAllErrorInfo();
        if (index == undefined) {
            let key = userName + "@" + ip + (port ? ":" + port : "");
            index = indexOfServer(key);
        }
        let cert = vue_object.serverData[index].cert;
        if (!cert.acceptedByUser) {
            cert = await GetCert(ip, port, index);
        } else {
            cert.ok = true;
        }

        if (!cert.pem) {
            return {
                success: false,
                index: index,
                cert: cert
            };
        }

        deleteFile(top.getSupportFile("commonResult", null, ip));
        let args = [];
        args.push(top.getSystemFile("oneCli"));
        args.push("ux");
        args.push("check-bmc-account");
        args.push("--bmc");
        args.push(getConnectionInfo(ip, port, userName));
        args.push("--never-check-trust");
        args.push("--noping");
        args.push(...add_output_log5(ip));
        top.store_onecli_command(args);
        printLog("INFO", "run commend:" + getRunProgramCommend(args));

        result = await runProgram(args);
        result.index = index;
        showOneCliErrorInfo(result, ip, index);
        log("scan {0} {1}".format(ip, result.success));
        if (result.success) {
            multi_config.AppendOrUpdate(ip, null, port, userName, null, null, null);
        }
        if (result.errorCode == PASSWORD_NEED_RESET && cert.ok) {
            vue_object.errorInfo = tr("The password needs to be changed");
            vue_object.errorInfoDetail = tr("It is the first time login or password has expired of current user, please modify the password.");
            vue_object.serverData[index].errorInfo = vue_object.errorInfo;
            vue_object.serverData[index].errorInfoDetail = vue_object.errorInfoDetail;
            result.success = true;
            vue_object.serverData[index].state = tr("Warning");
            vue_object.serverData[index].scanResult = tr("Fail");
            onFilterStringChanged();
            result.needChangePassword = true;
        }
        result.cert = cert;
        return result;
    }

    let onRowDel = function (rowIndex) {
        this.confirm = function (result) {
            vue_object.confirmDialogVisible = false;
            if (result) {
                if (vue_object.serverData[rowIndex].selected) {
                    vue_object.selectedCount--;
                }
                multi_config.RemoveItem(vue_object.serverData[rowIndex].ip);
                vue_object.serverData.splice(rowIndex, 1);
                checkNextEnable();
            }
        }
        vue_object.confirmDialogMessage = tr("Are you sure to delete this row - ") + vue_object.serverData[rowIndex].userName + "@" + vue_object.serverData[rowIndex].ip;
        vue_object.confirmDialogVisible = true;
    }

    let onRowEdit = function (rowIndex) {
        vue_object.isAllReady = false;
        vue_object.serverData[rowIndex].scanResult = "";
        vue_object.serverData[rowIndex].edit = true;
        vue_object.serverData[rowIndex].backUserName = vue_object.serverData[rowIndex].userName;
        vue_object.serverData[rowIndex].backPassword = vue_object.serverData[rowIndex].password;
    }

    let onRowEditFinished = async function (rowIndex) {
        vue_object.serverData[rowIndex].edit = false;
        vue_object.serverData[rowIndex].validateInfo = "";
        if (vue_object.serverData[rowIndex].backUserName === vue_object.serverData[rowIndex].userName && vue_object.serverData[rowIndex].backPassword === vue_object.serverData[rowIndex].password) {
            return;
        }
        if (vue_object.serverData[rowIndex].password === vue_object.serverData[rowIndex].backPassword) {
            vue_object.serverData[rowIndex].backUserName = vue_object.serverData[rowIndex].userName;
            multi_config.AppendOrUpdate(vue_object.serverData[rowIndex].ip
                , null, null, vue_object.serverData[rowIndex].userName);
        } else {
            await encrypt(vue_object.serverData[rowIndex].ip
                , vue_object.serverData[rowIndex].port
                , vue_object.serverData[rowIndex].userName
                , vue_object.serverData[rowIndex].password
                , rowIndex);
        }
        vue_object.serverData[rowIndex].scanResult = "";
    }

    let onRowRefresh = async function (rowIndex) {
        log("refresh server status ", rowIndex);
        vue_object.onecliIsRunning = true;
        findInfo.Total = 0;
        let ip = vue_object.serverData[rowIndex].ip;
        vue_object.scanMessage = tr("Scan {0} ...", ip);
        let port = vue_object.serverData[rowIndex].port;
        let userName = vue_object.serverData[rowIndex].userName;
        let password = vue_object.serverData[rowIndex].password;
        let encryptPassword = vue_object.serverData[rowIndex].value;
        if (!(userName || encryptPassword)) {
            dialog.credential.ip = ip;
            dialog.credential.port = port;
            dialog.credential.userName = userName;
            dialog.credential.password = password;
            let dlgResult = await dialog.credential.showDialog();
            if (!dlgResult) {
                vue_object.onecliIsRunning = false;
                return;
            }
            vue_object.serverData[rowIndex].userName = dialog.credential.userName;
            vue_object.serverData[rowIndex].password = dialog.credential.password;
            let result = await encrypt(ip, port, dialog.credential.userName, dialog.credential.password);
            if (!result) {
                vue_object.onecliIsRunning = false;
                return;
            }
        }        
        resetAllErrorInfo();
        let result = await doScan(ip, port, userName, rowIndex);
        vue_object.serverData[rowIndex].cert = result.cert;
        vue_object.serverData[rowIndex].needChangePassword = result.needChangePassword;
        if (!result.success) {
            vue_object.serverData[rowIndex].scanResult = tr("Fail");
            vue_object.serverData[rowIndex].state = tr("Error");
        } else {
            vue_object.serverData[rowIndex].scanResult = tr("Pass");
            vue_object.serverData[rowIndex].state = tr("Ready");
            vue_object.serverData[rowIndex].errorInfo = "";
        }
        if (!result.cert.ok && result.cert.pem) {
            vue_object.serverData[rowIndex].state = tr("Warning");
            vue_object.serverData[rowIndex].scanResult = tr("Fail");
        }
        if (result.needChangePassword) {
            vue_object.serverData[rowIndex].state = tr("Warning");
            vue_object.serverData[rowIndex].scanResult = tr("Fail");
        }
        if (!vue_object.errorInfo) {
            vue_object.errorInfo = vue_object.serverData[rowIndex].errorInfo;
        }
        if (!vue_object.errorInfoDetail) {
            vue_object.errorInfoDetail = vue_object.serverData[rowIndex].errorInfoDetail;
        }
        onFilterStringChanged(vue_object.filterString);
        vue_object.onecliIsRunning = false;
    }

    let onCheckAllChange = function (e) {
        log(vue_object.allChecked, e);
        for (var i = 0; i < vue_object.serverData.length; i++) {
            let value = vue_object.serverData[i];
            let lowerCaseKey = vue_object.filterString.toLowerCase();
            if ((value.mt.toLowerCase().includes(lowerCaseKey))
                || (value.ip.toLowerCase().includes(lowerCaseKey))
                || (value.sn.toLowerCase().includes(lowerCaseKey))
                || (value.userName.toLowerCase().includes(lowerCaseKey))
            ) {
                value.selected = vue_object.allChecked;
            }
        }
        onCheckItemChange();
    }

    let onCheckItemChange = function (rowIndex) {
        let allseled = true;
        let allUnSeled = true;
        vue_object.selectedCount = 0;
        for (var i = 0; i < vue_object.serverData.length; i++) {
            let value = vue_object.serverData[i];
            if (value.selected) {
                allUnSeled = false;
                vue_object.selectedCount++;
            } else {
                allseled = false;
            }
        }
        if (!allseled && !allUnSeled) {
            if (vue_object.checkAllIndeterminate) {
                vue_object.checkAllIndeterminate = false;
                setTimeout(function () { vue_object.checkAllIndeterminate = true }, 1);
            } else {
                vue_object.checkAllIndeterminate = true;
                vue_object.allChecked = true;
            }
        } else if (allseled && vue_object.serverData.length > 0) {
            vue_object.checkAllIndeterminate = false;
            vue_object.allChecked = true;
        } else {
            vue_object.checkAllIndeterminate = false;
            vue_object.allChecked = false;
        }
        log(vue_object.checkAllIndeterminate);
        checkNextEnable();
    }

    let rowClick = function (index) {
        log(index)
    }

    let cancelOnecli = function () {
        for (let item of multiRunning.idList) {
            clearTimeout(item);
        }
        multiRunning.idList.clear();
        multiRunning.delayCount = 0;
        log("cancel onecli");
        top.killOneCli();
    }

    let getConnectionInfo = function (ip, port, userName) {
        if (ip.includes(":")) {
            if (port && port != "") {
                return (userName + "@[" + ip + "]:" + port);
            } else {
                return (userName + "@[" + ip + "]");
            }
        } else {
            if (port && port != "") {
                return (userName + "@" + ip + ":" + port);
            } else {
                return (userName + "@" + ip);
            }
        }
    }

    async function checkEdit() {
        for (var i = 0; i < vue_object.serverData.length; i++) {
            if (vue_object.serverData[i].selected) {
                if (vue_object.serverData[i].edit) {
                    vue_object.serverData[i].edit = false;
                    await onRowEditFinished(i);
                }
            }
        }
    }

    let checkAllState = async function () {
        await checkEdit();
        resetAllErrorInfo();        
        vue_object.isAllReady = await onScanSelected();
        let getReadyState = function () {
            let ready = true;
            for (var i = 0; i < vue_object.serverData.length; i++) {
                let item = vue_object.serverData[i];
                if (item.selected) {
                    if (item.state != tr("Ready")) {
                        ready = false;
                    }
                }
            }
            return ready;
        }
        vue_object.isAllReady = getReadyState();
        if (!vue_object.isAllReady) {
            let existCertWarning = false;
            for (var i = 0; i < vue_object.serverData.length; i++) {
                let item = vue_object.serverData[i];
                if (!item.cert.pem) {
                    continue;
                }
                if (!item.cert.ok && !item.cert.acceptedByUser) {
                    existCertWarning = true;
                    break;
                }
            }
            if (existCertWarning) {
                let result = false;
                result = await dialog.batchCertificate.showDialog();
                if (result) {
                    gl_wizard.steps("next");
                    return;
                }
                return;
            }
            let val = await dialog.changePasswordBatch.check();
            if (!val) {
                return;
            }
            gl_wizard.steps("next");
        }
        if (vue_object.isAllReady) {
            gl_wizard.steps("next");
        }
        return true;
    }

    let onPasswordVisibleClicked = function () {
        vue_object.passwordIsVisible = !vue_object.passwordIsVisible
        setTimeout(function () {
            $("#server-pool-input-passwor").focus();
        }, 200);
    }

    let onImport = async function () {
        printLog("INFO", "button 'Import...' was clicked.");
        let _filterType = [{ name: 'Json File', extensions: ["json"] }];
        var path = window.xpress.showOpenDialogFile("configure.json", _filterType);
        if (path && path[0]) {
            var dir = path[0];
            printLog("INFO", "The file was selected: ", dir);
            log(dir);
            resetAllErrorInfo();
            var result = await multi_config.Merge(dir);
            if (!result) {
                showOneCliErrorInfo({
                    success: false,
                    stderr: ""
                }, "", -1, tr("Import configure file failed."));
            } else {
                vue_object.statusInfo = "Import from file <i>{0}</i> success!".format(dir);
            }
            refresh();
        }
    }

    let onExport = async function () {
        printLog("INFO", "button 'Export...' was clicked.");
        resetAllErrorInfo();
        let _filterType = [
            { name: 'Excel File', extensions: ["csv"] },
            { name: 'Excel File', extensions: ["xls"] },
            { name: 'Json File', extensions: ["json"] }
        ];
        var path = window.xpress.showSaveDialog(exportData.fileName, _filterType);
        if (path) {
            printLog("INFO", "The file was selected: ", path);
            log(path);
            exportData.fileName = path;
            var result = false;
            if (path.toLowerCase().endsWith(".json")) {
                result = multi_config.Export(path);
            } else {
                result = await onExportToExcel();
            }
            printLog("Export to file: ", result);
            log("Export to file: ", result);
            if (result == true) {
                vue_object.statusInfo = "Export to file <i>{0}</i> {1}!".format(path, "success");
            } else {
                vue_object.errorInfo = result;
            }
        }
    }

    let onScanSelected = async function () {
        findInfo.reset();
        multiRunning.reset();
        vue_object.errorInfo = "";
        vue_object.errorInfoDetail = "";
        vue_object.scanMessage = "Scanning ...";
        if (vue_object.selectedCount == 0) {
            vue_object.errorInfo = "";
            setTimeout(function () {
                vue_object.errorInfo = tr("Warning: No servers are selected, please check it.");
            }, 100);
            return false;
        }
        vue_object.onecliIsRunning = true;
        let unAuthCount = 0;
        for (var i = 0; i < vue_object.serverData.length; i++) {
            if (vue_object.serverData[i].selected) {
                if (!(vue_object.serverData[i].userName || vue_object.serverData[i].password)) {
                    unAuthCount++;
                }
            }
        }
        if (unAuthCount > 0) {
            dialog.credential.ip = "***.***.***.***";
            dialog.credential.port = '';
            dialog.credential.userName = "";
            dialog.credential.password = "";
            let dlgResult = await dialog.credential.showDialog();
            if (!dlgResult) {
                vue_object.onecliIsRunning = false;
                return false;
            }
            for (var i = 0; i < vue_object.serverData.length; i++) {
                let ip = vue_object.serverData[i].ip;
                let port = vue_object.serverData[i].port;
                if (vue_object.serverData[i].selected) {
                    if (!(vue_object.serverData[i].userName || vue_object.serverData[i].password)) {
                        log("encrypt:", ip);
                        let val = await encrypt(ip, port,
                            dialog.credential.userName, dialog.credential.password, i);
                        if (!val) {
                            continue;
                        }
                        vue_object.serverData[i].userName = dialog.credential.userName;
                        vue_object.serverData[i].password = dialog.credential.password;
                    }
                }
            }
        }
        

        let updateItem = function (result, item) {
            item.scanResult = result.success ? tr("Pass") : tr("Fail");
            item.cert = result.cert;
            item.needChangePassword = result.needChangePassword;
            multi_config.AppendOrUpdate(item.ip, null, null, null, null, null, null);
            if (!item.cert.pem) {
                item.state = tr("Error");
                item.scanResult = tr("Fail");
            }
            else if ((!item.cert.ok && !item.cert.acceptedByUser) || result.needChangePassword) {
                item.state = tr("Warning");
                item.scanResult = tr("Fail");
            }
            else if (result.success) {
                item.state = tr("Ready");
                item.errorInfo = "";
                item.scanResult = tr("Pass");
            } else {
                item.state = tr("Error");
                item.scanResult = tr("Fail");
            }
            onFilterStringChanged();
        }
        
        let scanProcess = [];

        let allSuccess = true;

        for (let item of vue_object.serverData) {
            if (item.selected) {
                multiRunning.scanningTotal++;
                if (item.edit) {
                    item.edit = true;
                    await onRowEditFinished(vue_object.serverData.indexOf(item));
                }
                let promise = doScan(item.ip, item.port, item.userName);
                scanProcess.push({
                    item: item,
                    promise: promise
                });
                if (scanProcess.length > 1) {
                    let result = await scanProcess[0].promise;
                    allSuccess &= result.success;
                    allSuccess &= !result.needChangePassword;
                    updateItem(result, scanProcess[0].item);
                    scanProcess.splice(0, 1);
                }
            }
        }

        for (let item of scanProcess) {
            let result = await item.promise;
            allSuccess &= result.success;
            allSuccess &= !result.needChangePassword;
            updateItem(result, item.item);
        }
        vue_object.onecliIsRunning = false;
        return allSuccess;
    }

    let onMouseout = function (ctl) {
        if (document.activeElement == ctl.target) {
            return;
        }
        tipsInfo = '';
    }

    let onRemoveSelected = function () {
        if (vue_object.selectedCount == 0) {
            vue_object.errorInfo = "";
            setTimeout(function () {
                vue_object.errorInfo = tr("Warning: No servers are selected, please check it.");
            }, 100);
            return;
        }
        this.confirm = function (result) {
            vue_object.confirmDialogVisible = false;
            if (result) {
                for (let i = 0; i < vue_object.serverData.length;) {
                    if (vue_object.serverData[i].selected) {
                        multi_config.RemoveItem(vue_object.serverData[i].ip);
                        vue_object.serverData.splice(i, 1);
                        continue;
                    }
                    i++;
                }
                checkNextEnable();
            }
        }
        vue_object.confirmDialogMessage = tr("Are you sure to delete all the selected rows");
        vue_object.confirmDialogVisible = true;
    }

    let onIPSuffixChanged = function (ctl) {
        let tempVal = bmcInfo.ipFrom;
        let suffix = 0;
        let val = parseInt(ctl);
        if (top.isIPV4(tempVal)) {
            let index = tempVal.lastIndexOf(".");
            suffix = parseInt(tempVal.substring(index + 1))
            if (suffix > val) {
                bmcInfo.ipSuffix = suffix;
                vue_object.errorInfo = tr("The IP address in 'To' must be bigger or equal than 'From'")
            } else if (val > 255) {
                bmcInfo.ipSuffix = 255;
            }
        } else if (top.isIPV6(tempVal)) {
            let index = tempVal.lastIndexOf(":");
            suffix = tempVal.substring(index + 1);
        }
    }

    let onPortBlur = function (ctl) {
        let val = parseInt(ctl.target.value);
        if (ctl.target.value == "") {
            vue_object.tipsInfo = "";
            return;
        }
        if (isNaN(val) || val < 1 || val > 65535) {
            ctl.target.focus();
        } else {
            vue_object.tipsInfo = "";
        }
    }

    let onRowConfirmCert = function (index) {
        resetAllErrorInfo();
        vue_object.errorInfo = vue_object.serverData[index].errorInfo;
        dialog.bmcCertificate.certText = vue_object.serverData[index].cert.pem;
        dialog.bmcCertificate.ip = vue_object.serverData[index].ip;
        dialog.bmcCertificate.port = vue_object.serverData[index].port;
        dialog.bmcCertificate.userName = vue_object.serverData[index].userName;
        dialog.bmcCertificate.index = index;
        dialog.bmcCertificate.showDialog();
    }

    let onChangePassword = async function (index) {
        resetAllErrorInfo();
        vue_object.errorInfo = vue_object.serverData[index].errorInfo;
        vue_object.errorInfoDetail = vue_object.serverData[index].errorInfoDetail;
        dialog.changePassword.ip = vue_object.serverData[index].ip;
        dialog.changePassword.port = vue_object.serverData[index].port;
        dialog.changePassword.userName = vue_object.serverData[index].userName;
        dialog.changePassword.index = index;
        return dialog.changePassword.showDialog();
    }

    let onShowItemFailedInfo = function (index) {
        resetAllErrorInfo();
        setTimeout(() => {
            vue_object.errorInfo = vue_object.serverData[index].errorInfo;
            vue_object.errorInfoDetail = vue_object.serverData[index].errorInfoDetail;
        }, 200);
    }

    function onSelectionChanged(context) {
        vue_object.selectedCount = 0;
        for (var item of vue_object.serverData) {
            item.selected = false;
        }
        for (var item of context) {
            item.selected = true;
            vue_object.selectedCount++;
        }
    }

    let onFilterStringChanged = function (val) {
        if (!val) val = vue_object.filterString;
        vue_object.serverDataDisplay = [];
        for (let item of vue_object.serverData) {
            let needShow = false;
            let newVal = val.toUpperCase();
            if (item.ip.toUpperCase().includes(newVal)) {
                item.ipDisplay = item.ip.toUpperCase().replace(newVal, "<b style='color:red'>" + val + "</b>");
                needShow = true;
            } else {
                item.ipDisplay = item.ip;
            }
            if (item.mt.toUpperCase().includes(newVal)) {
                let regex = new RegExp(newVal, "ig");
                let tempVal = item.mt.match(regex);
                item.mtDisplay = item.mt.toUpperCase().replace(regex, "<b style='color:red'>" + tempVal[0] + "</b>");
                needShow = true;
            } else {
                item.mtDisplay = item.mt;
            }
            if (item.sn.toUpperCase().includes(newVal)) {
                let regex = new RegExp(newVal, "ig");
                let tempVal = item.sn.match(regex);
                item.snDisplay = item.sn.toUpperCase().replace(regex, "<b style='color:red'>" + tempVal[0] + "</b>");
                needShow = true;
            } else {
                item.snDisplay = item.sn;
            }
            if (item.state.toUpperCase().includes(newVal)) {
                let regex = new RegExp(newVal, "ig");
                let tempVal = item.state.match(regex);
                item.stateDisplay = item.state.replace(regex, "<b style='color:red'>" + tempVal[0] + "</b>");
                needShow = true;
            }
            if (needShow) {
                vue_object.serverDataDisplay.push(item);
            }
        }
    }

    var tableRef = Vue.ref();

    vue_class = {
        data() {
            return {
                vue_object: vue_object,
                dialog: dialog,
                ui: ui,
                bmcInfo: bmcInfo,
                findInfo: findInfo,
                multiRunning: multiRunning,
                rules: rules,
                exportData: exportData,
                tableRef: tableRef
            };
        },
        methods: {
            onRowRefresh: onRowRefresh,
            onRowDel: onRowDel,
            onShowItemFailedInfo: onShowItemFailedInfo,
            onChangePassword: onChangePassword,
            onScanSelected: onScanSelected,
            onRemoveSelected: onRemoveSelected,
            onRowConfirmCert: onRowConfirmCert,
            cancelOnecli: cancelOnecli,
            onDiscover: onAddByOnecli,
            onMouseout: onMouseout,
            onPortBlur: onPortBlur,
            onIPSuffixChanged: onIPSuffixChanged,
            onImport: onImport,
            onExport: onExport,
            onExportToExcel: onExportToExcel,
            onCheckAllChange: onCheckAllChange,
            onPasswordVisibleClicked: onPasswordVisibleClicked,
            resetAllErrorInfo: resetAllErrorInfo,
            onRowEdit: onRowEdit,
            onRowEditFinished: onRowEditFinished,
            onCheckItemChange: onCheckItemChange,
            onSelectionChanged: onSelectionChanged,
            tr: top.tr
        },
        mounted() {
            $("#server-pool-dialogs").show();
        },
        watch: {
            'vue_object.filterString': onFilterStringChanged,
            "bmcInfo.ipFrom": function (val) {
                if (top.isIPV4(val)) {
                    bmcInfo.ipFromFormatError = false;
                    let index = val.lastIndexOf(".");
                    bmcInfo.ipPrefix = val.substring(0, index + 1);
                    if (!bmcInfo.ipSuffix
                        || (bmcInfo.ipSuffix
                            && parseInt(bmcInfo.ipSuffix) < parseInt(val.substring(index + 1))
                        )
                    ) {
                        bmcInfo.ipSuffix = val.substring(index + 1);
                    } else {
                        bmcInfo.ipSuffix = "";
                    }
                } else if (top.isIPV6(val)) {
                    bmcInfo.ipFromFormatError = false;
                    let index = val.lastIndexOf(":");
                    bmcInfo.ipPrefix = val.substring(0, index + 1);
                    if (!bmcInfo.ipSuffix
                        || (bmcInfo.ipSuffix
                            && parseInt(bmcInfo.ipSuffix) < parseInt(val.substring(index + 1))
                        )
                    ) {
                        bmcInfo.ipSuffix = val.substring(index + 1);
                    } else {
                        bmcInfo.ipSuffix = "";
                    }
                } else {
                    bmcInfo.ipFromFormatError = true;
                    bmcInfo.ipPrefix = "";
                    bmcInfo.ipSuffix = "";
                }
            },
            "multiRunning.runningCount": function (val) {
                if (val == 0 && findInfo && findInfo.total > 0) {
                    vue_object.scanResultInfo = tr("Total scanned:") + " " + findInfo.total
                        + ", <span style='color:darkgreen'>"
                        + findInfo.passCount + " " + tr("matched") + "</span>"
                        + ", <span style='color:chocolate'>" + findInfo.failCount + " " + tr("skipped") + "</span>";
                }
            },
            "vue_object.selectedCount": function (val) {
                vue_object.isAllReady = false;
                top.toggleNext(val > 0);
            },
            "exportData.loading": function (val) {
                if (val) {
                    this.loading = ElementPlus.ElLoading.service({
                        lock: true,
                        text: 'Reading...',
                        background: 'rgba(0, 0, 0, 0.7)',
                    });
                    return;
                }
                this.loading.close();
            }
        },
        directives: {
            focus: {
                mounted(el) {
                    el.focus();
                }
            }
        },
        computed: {
            discover() {
                if (!top.ipValidate(bmcInfo.ipFrom)) {
                    return true;
                }
                if (ui.credential_show) {
                    if (bmcInfo.userName === "") {
                        return true;
                    }
                    if (bmcInfo.password === "") {
                        return true;
                    }
                }
                return false;
            }
        }

    }

    return {
        index: 0,
        name: "Server_Pool",
        init: init,
        nextAction: next,
        previousAction: previous,
        importId: "#import_server_pool",
        selectorId: ".server_pool",
        title: "Navi.ServerPool",
        localizedTitle: tr("Navi.ServerPool"),
        uninit: uninit,
        selectedServerList: []
    }
}();
