param (
    [switch]$Verbose
)

$ScriptPath   = $MyInvocation.MyCommand.Path
$SettingsPath = $ScriptPath -replace '\.ps1$', '.ini'
$Settings     = Get-Content "$SettingsPath" | ForEach-Object {
    if ($_ -match '^(.+?)=(.+)') {
        [PsCustomObject]@{
            Name = $Matches[1].Trim()
            Value = $Matches[2].Trim().Trim('"')
        }
    }
}

$ScriptLog    = $ScriptPath -replace '\.ps1$','.log'

$SourceFiles  = ($Settings | Where-Object {$_.Name -like 'File*'}).Value
$RefreshToken = ($Settings | Where-Object {$_.Name -eq 'RefreshToken'}).Value
$ClientID     = ($Settings | Where-Object {$_.Name -eq 'ClientID'}).Value
$ClientSecret = ($Settings | Where-Object {$_.Name -eq 'ClientSecret'}).Value
$ParentId     = ($Settings | Where-Object {$_.Name -eq 'ParentId'}).Value

$MaxLogFileLines = 1000

if ($Verbose) {
    $variablesTable = @{
        ScriptPath = $ScriptPath
        SettingsPath = $SettingsPath
        ScriptLog = $ScriptLog
        SourceFiles = $SourceFiles -join ', '
        RefreshToken = $RefreshToken
        ClientID = $ClientID
        ClientSecret = $ClientSecret
        ParentId = $ParentId
        MaxLogFileLines = $MaxLogFileLines
    }

    $variablesTable | Format-Table
    exit
}

# Function to write log entries
function Write-Log {
    param(
        [string]$Type,
        [string]$Message
    )

    $logEntry = "$(Get-Date -Format 'yyyy-MM-dd'),$(Get-Date -Format 'HH:mm:ss'),$Type,$Message"
    $logEntry | Out-File -FilePath $ScriptLog -Encoding UTF8 -Append
    Write-Host $logEntry

    # Trim log file to the maximum number of lines
    $logLines = Get-Content -Path $ScriptLog
    if ($logLines.Count -gt $MaxLogFileLines) {
        $logLines | Select-Object -Last $MaxLogFileLines | Set-Content -Path $ScriptLog
    }
}

Write-Log -Type 'START' -Message 'Script started.'

foreach ($SourceFile in $SourceFiles) {
    $SourceFileHash = "$SourceFile.hash"
    $previousHash = ''
    if (Test-Path $SourceFileHash) {
        $previousHash = Get-Content $SourceFileHash
    }

    $currentHash = Get-FileHash -Path $SourceFile | Select-Object -ExpandProperty Hash

    if ($currentHash -eq $previousHash) {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is the same. No need to upload."
        continue
    } else {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is different. Proceeding with the upload."
        Set-Content -Path $SourceFileHash -Value $currentHash
    }

    try {
        # Authenticate and authorize
        $authParams = @{
            'refresh_token' = $RefreshToken
            'client_id'     = $ClientID
            'client_secret' = $ClientSecret
            'grant_type'    = 'refresh_token'
        }
        $accessTokenResponse = Invoke-RestMethod -Uri 'https://www.googleapis.com/oauth2/v4/token' -Method POST -Body $authParams
        $accessToken = $accessTokenResponse.access_token

        # Check if the file exists in Google Drive
		$encodedSourceFile = [System.Uri]::EscapeDataString((Get-Item $SourceFile).Name)
		$query = "name='$encodedSourceFile' and trashed=false"
		$fileResponse = Invoke-RestMethod -Uri "https://www.googleapis.com/drive/v3/files?q=$query" -Method GET -Headers @{ 'Authorization' = "Bearer $accessToken" }
        $existingFile = $fileResponse.files | Select-Object -First 1
		Write-Log -Type 'INFO' -Message "Files in google drive: '$existingFile'"

        if ($existingFile) {
            # File already exists, delete it
            Invoke-RestMethod -Uri "https://www.googleapis.com/drive/v3/files/$($existingFile.id)" -Method DELETE -Headers @{ 'Authorization' = "Bearer $accessToken" }
			Write-Log -Type 'INFO' -Message "File '$SourceFile' existed and was deleted."
        }

        # Upload the file
		$encodedFileName = [System.Uri]::EscapeDataString((Get-Item $SourceFile).Name)
		$uploadParams = @{
			'name'            = $encodedFileName
			'parents'         = @($ParentId)
			'uploadType'      = 'media'
			'access_token'    = $accessToken
			'Content-Type'    = 'application/octet-stream'
			'InFile'          = Get-Item $SourceFile
		}
        $uploadUri = "https://www.googleapis.com/upload/drive/v3/files?$(foreach ($param in $uploadParams.GetEnumerator()) { "$($param.Name)=$($param.Value)" -replace '\s', '%20' })"
        $uploadResponse = Invoke-RestMethod -Uri $uploadUri -Method POST -InFile $SourceFile -Headers @{ 'Authorization' = "Bearer $accessToken" }

        Write-Log -Type 'INFO' -Message "Upload for '$SourceFile' completed."
    }
    catch {
        Write-Log -Type 'ERROR' -Message $_.Exception.Message
        continue
    }
}

Write-Log -Type 'STOP' -Message 'Script stopped.'
