param (
    [switch]$Verbose
)

$ScriptPath   = $MyInvocation.MyCommand.Path
$SettingsPath = $ScriptPath -replace '\.ps1$', '.ini'
$Settings     = Get-Content "$SettingsPath" | ForEach-Object {
    if ($_ -match '^(.+?)=(.+)') {
        [PsCustomObject]@{
            Name = $Matches[1].Trim()
            Value = $Matches[2].Trim().Trim('"')
        }
    }
}

$ScriptLog    = $ScriptPath -replace '\.ps1$','.log'

$SourceFile   = ($Settings | Where-Object {$_.Name -eq 'File2Upload'}).Value
$RefreshToken = ($Settings | Where-Object {$_.Name -eq 'RefreshToken'}).Value
$ClientID     = ($Settings | Where-Object {$_.Name -eq 'ClientID'}).Value
$ClientSecret = ($Settings | Where-Object {$_.Name -eq 'ClientSecret'}).Value
$ParentId     = ($Settings | Where-Object {$_.Name -eq 'ParentId'}).Value

$MaxLogFileLines = 1000

if ($Verbose) {
    $variablesTable = @{
        ScriptPath = $ScriptPath
        SettingsPath = $SettingsPath
        ScriptLog = $ScriptLog
        SourceFile = $SourceFile
        RefreshToken = $RefreshToken
        ClientID = $ClientID
        ClientSecret = $ClientSecret
        ParentId = $ParentId
        MaxLogFileLines = $MaxLogFileLines
    }

    $variablesTable | Format-Table
    exit
}

if (-not (Test-Path $ScriptLog)) {
    $logHeader = 'Date,Time,Type,Message'
    $logHeader | Out-File -FilePath $ScriptLog -Encoding UTF8
    Write-Host "The log file '$ScriptLog' was created."
}

function Write-Log {
    param([string]$Type, [string]$Message)
    $logEntry = "$(Get-Date -Format 'yyyy-MM-dd'),$(Get-Date -Format 'HH:mm:ss'),$Type,$Message"
    $logEntry | Out-File -FilePath $ScriptLog -Encoding UTF8 -Append
    Write-Host $logEntry

    $logLines = Get-Content -Path $ScriptLog
    if ($logLines.Count -gt $MaxLogFileLines) {
        $logLines | Select-Object -Last $MaxLogFileLines | Set-Content -Path $ScriptLog
    }
}

Write-Log -Type 'START' -Message 'Script started.'

$SourceFileHash = "$SourceFile.hash"
$previousHash = ''
if (Test-Path $SourceFileHash) {
    $previousHash = Get-Content $SourceFileHash
}

$currentHash = Get-FileHash -Path $SourceFile | Select-Object -ExpandProperty Hash

if ($currentHash -eq $previousHash) {
    Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is the same. No need to upload."
    return
} else {
    Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is different. Proceeding with the upload."
    Set-Content -Path $SourceFileHash -Value $currentHash
}

$accessTokenParams = @{
    Uri = 'https://accounts.google.com/o/oauth2/token'
    Body = "refresh_token=$RefreshToken&client_id=$ClientID&client_secret=$ClientSecret&grant_type=refresh_token"
    Method = 'Post'
    ContentType = 'application/x-www-form-urlencoded'
}

try {
    $accessTokenResponse = Invoke-RestMethod @accessTokenParams
    $accessToken = $accessTokenResponse.access_token

    $sourceBase64 = [Convert]::ToBase64String([IO.File]::ReadAllBytes((Get-Item $SourceFile).FullName))
    $sourceMime = [System.Web.MimeMapping]::GetMimeMapping((Get-Item $SourceFile).FullName)
    $supportsTeamDrives = 'false'

    $uploadMetadata = @{
        originalFilename = (Get-Item $SourceFile).Name
        name = (Get-Item $SourceFile).Name
        parents = @($ParentId)
    }

    $uploadBody = @"
--boundary
Content-Type: application/json; charset=UTF-8

$($uploadMetadata | ConvertTo-Json)

--boundary
Content-Transfer-Encoding: base64
Content-Type: $sourceMime

$sourceBase64
--boundary--
"@

    $uploadHeaders = @{
        "Authorization" = "Bearer $accessToken"
        "Content-Type" = 'multipart/related; boundary=boundary'
        "Content-Length" = $uploadBody.Length
    }

    $uploadParams = @{
        Uri = "https://www.googleapis.com/upload/drive/v3/files?uploadType=multipart&supportsTeamDrives=$supportsTeamDrives"
        Method = 'Post'
        Headers = $uploadHeaders
        Body = $uploadBody
    }

    $response = Invoke-RestMethod @uploadParams

    Write-Log -Type 'Info' -Message "Upload for '$SourceFile' completed."
} catch {
    Write-Log -Type 'Error' -Message $_.Exception.Message
}

Write-Log -Type 'STOP' -Message 'Script stopped.'
