param (
    [switch]$Verbose
)

$ScriptPath   = $MyInvocation.MyCommand.Path
$SettingsPath = $ScriptPath -replace '\.ps1$', '.ini'
$Settings     = Get-Content "$SettingsPath" | ForEach-Object {
    if ($_ -match '^(.+?)=(.+)') {
        [PsCustomObject]@{
            Name = $Matches[1].Trim()
            Value = $Matches[2].Trim().Trim('"')
        }
    }
}

$ScriptLog    = $ScriptPath -replace '\.ps1$','.log'

$SourceFiles  = ($Settings | Where-Object {$_.Name -like 'File*'}).Value
$RefreshToken = ($Settings | Where-Object {$_.Name -eq 'RefreshToken'}).Value
$ClientID     = ($Settings | Where-Object {$_.Name -eq 'ClientID'}).Value
$ClientSecret = ($Settings | Where-Object {$_.Name -eq 'ClientSecret'}).Value
$ParentId     = ($Settings | Where-Object {$_.Name -eq 'ParentId'}).Value

$MaxLogFileLines = 1000

if ($Verbose) {
    $variablesTable = @{
        ScriptPath = $ScriptPath
        SettingsPath = $SettingsPath
        ScriptLog = $ScriptLog
        SourceFiles = $SourceFiles -join ', '
        RefreshToken = $RefreshToken
        ClientID = $ClientID
        ClientSecret = $ClientSecret
        ParentId = $ParentId
        MaxLogFileLines = $MaxLogFileLines
    }

    $variablesTable | Format-Table
    exit
}

# Function to write log entries
function Write-Log {
    param(
        [string]$Type,
        [string]$Message
    )

    $logEntry = "$(Get-Date -Format 'yyyy-MM-dd'),$(Get-Date -Format 'HH:mm:ss'),$Type,$Message"
    $logEntry | Out-File -FilePath $ScriptLog -Encoding UTF8 -Append
    Write-Host $logEntry

    # Trim log file to the maximum number of lines
    $logLines = Get-Content -Path $ScriptLog
    if ($logLines.Count -gt $MaxLogFileLines) {
        $logLines | Select-Object -Last $MaxLogFileLines | Set-Content -Path $ScriptLog
    }
}

Write-Log -Type 'START' -Message 'Script started.'

foreach ($SourceFile in $SourceFiles) {
    $SourceFileHash = "$SourceFile.hash"
    $previousHash = ''
    if (Test-Path $SourceFileHash) {
        $previousHash = Get-Content $SourceFileHash
    }

    $currentHash = Get-FileHash -Path $SourceFile | Select-Object -ExpandProperty Hash

    if ($currentHash -eq $previousHash) {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is the same. No need to upload."
        continue
    } else {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is different. Proceeding with the upload."
        Set-Content -Path $SourceFileHash -Value $currentHash
    }

    $accessTokenParams = @{
        Uri = 'https://accounts.google.com/o/oauth2/token'
        Body = "refresh_token=$RefreshToken&client_id=$ClientID&client_secret=$ClientSecret&grant_type=refresh_token"
        Method = 'Post'
        ContentType = 'application/x-www-form-urlencoded'
    }

    try {
        $accessTokenResponse = Invoke-RestMethod @accessTokenParams
        $accessToken = $accessTokenResponse.access_token

        $sourceBase64 = [Convert]::ToBase64String([IO.File]::ReadAllBytes((Get-Item $SourceFile).FullName))
        $sourceMime = [System.Web.MimeMapping]::GetMimeMapping((Get-Item $SourceFile).FullName)
        $supportsTeamDrives = 'false'

        # Check if the file exists in Google Drive
        $fileParams = @{
            Uri = "https://www.googleapis.com/drive/v3/files"
            Method = 'Get'
            Headers = @{
                "Authorization" = "Bearer $accessToken"
            }
            ContentType = 'application/json'
            Body = @{
                "q" = "name='$SourceFile' and trashed=false"
                "supportsAllDrives" = $supportsTeamDrives
            } | ConvertTo-Json
        }

        $fileResponse = Invoke-RestMethod @fileParams

        if ($fileResponse.files) {
            # File already exists, update it
            $fileId = $fileResponse.files[0].id
            $uploadUri = "https://www.googleapis.com/upload/drive/v3/files/$fileId?uploadType=multipart&supportsAllDrives=$supportsTeamDrives"
            $uploadMethod = 'Patch'
        } else {
            # File doesn't exist, create it
            $uploadUri = "https://www.googleapis.com/upload/drive/v3/files?uploadType=multipart&supportsAllDrives=$supportsTeamDrives"
            $uploadMethod = 'Post'
        }

        $uploadMetadata = @{
            name = (Get-Item $SourceFile).Name
            parents = @($ParentId)
        }

        $uploadBody = @"
--boundary
Content-Type: application/json; charset=UTF-8

$($uploadMetadata | ConvertTo-Json)

--boundary
Content-Transfer-Encoding: base64
Content-Type: $sourceMime

$sourceBase64
--boundary--
"@

        $httpClient = New-Object System.Net.Http.HttpClient
        $httpClient.DefaultRequestHeaders.Authorization = New-Object System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", $accessToken)
        $content = New-Object System.Net.Http.ByteArrayContent([System.Text.Encoding]::UTF8.GetBytes($uploadBody))
        $content.Headers.ContentType = New-Object System.Net.Http.Headers.MediaTypeHeaderValue("multipart/related; boundary=boundary")

        $uploadResponse = $httpClient.$uploadMethod($uploadUri, $content)

        Write-Log -Type 'INFO' -Message "Upload for '$SourceFile' completed."
    } catch {
        Write-Log -Type 'ERROR' -Message $_.Exception.Message
        continue
    }
}

Write-Log -Type 'STOP' -Message 'Script stopped.'
